<?php

namespace App\Services;


use App\Models\Fund;
use App\Models\PrepaidCredit;
use App\Models\SubscriptionPurchase;
use App\Models\Transaction;
use App\Traits\Notify;
use Carbon\Carbon;

class BasicService
{
    use Notify;

    public function setEnv($value)
    {
        $envPath = base_path('.env');
        $env = file($envPath);
        foreach ($env as $env_key => $env_value) {
            $entry = explode("=", $env_value, 2);
            $env[$env_key] = array_key_exists($entry[0], $value) ? $entry[0] . "=" . $value[$entry[0]] . "\n" : $env_value;
        }
        $fp = fopen($envPath, 'w');
        fwrite($fp, implode($env));
        fclose($fp);
    }

    public function preparePaymentUpgradation($deposit): void
    {
        $basicControl = basicControl();

        try {
            if ($deposit->status == 0 || $deposit->status == 2) {
                if ($deposit->depositable_type == SubscriptionPurchase::class) {
                    $subscriptionPurchase = $deposit->depositable;
                    $subscriptionPurchase->payment_status = 1;
                    $subscriptionPurchase->save();

                    $user = $deposit->user;
                    $user->limit_contact = $subscriptionPurchase->number_of_contacts;
                    $user->limit_emails = $subscriptionPurchase->number_of_emails;
                    $user->plan_id = $subscriptionPurchase->id;
                    $user->save();

                    if ($subscriptionPurchase->frequency == 'monthly') {
                        $user->subs_expired_at = Carbon::now()->addDays(30);
                        $subscriptionPurchase->subs_expired_at = Carbon::now()->addDays(30);
                    } elseif ($subscriptionPurchase->frequency == 'yearly') {
                        $user->subs_expired_at = Carbon::now()->addDays(365);
                        $subscriptionPurchase->subs_expired_at = Carbon::now()->addDays(365);
                    }

                    $user->save();
                    $subscriptionPurchase->save();

                    $fund = $this->newFund($deposit);
                    $this->newTransaction($fund, 'Subscription Via');

                    $deposit->save();
                    $params = [
                        'amount' => getAmount($deposit->amount, $basicControl->fraction_number),
                        'currency' => $basicControl->base_currency,
                        'planName' => $subscriptionPurchase->plan_name ?? null,
                        'transaction' => $deposit->utr,
                    ];

                    $action = [
                        "link" => "#",
                        "icon" => "fa fa-money-bill-alt text-white"
                    ];

                    $this->sendMailSms($deposit->receiver, 'USER_BUY_PLAN', $params);
                    $this->userPushNotification($deposit->receiver, 'USER_BUY_PLAN', $params, $action);
                    $firebaseAction = route('user.fund.index');
                    $this->userFirebasePushNotification($deposit->receiver, 'USER_BUY_PLAN', $params, $firebaseAction);

                    $params = [
                        'user' => optional($deposit->receiver)->username ?? null,
                        'amount' => getAmount($deposit->amount, $basicControl->fraction_number),
                        'currency' => $basicControl->base_currency,
                        'planName' => $subscriptionPurchase->plan_name ?? null,
                        'transaction' => $deposit->utr,
                    ];

                    $action = [
                        "link" => '#',
                        "icon" => "fa fa-money-bill-alt text-white"
                    ];

                    $this->adminMail('USER_BUY_PLAN_ADMIN', $params);
                    $this->adminPushNotification('USER_BUY_PLAN_ADMIN', $params, $action);
                    $this->adminFirebasePushNotification('USER_BUY_PLAN_ADMIN', $params);

                } elseif ($deposit->depositable_type == PrepaidCredit::class) {
                    $prepaidCredit = $deposit->depositable;
                    $user = $deposit->user;
                    $user->credits += $prepaidCredit->credits;
                    $user->save();

                    $fund = $this->newFund($deposit);
                    $this->newTransaction($fund, 'Prepaid Credits Via');
                }

                $deposit->status = 1;
                $deposit->save();

            }
        } catch (\Exception $e) {
        }
    }

    public function subscriptionUpgrade($deposit)
    {
        $basicControl = basicControl();
        if ($deposit->depositable_type == SubscriptionPurchase::class) {
            $subscriptionPurchase = $deposit->depositable;
            $subscriptionPurchase->payment_status = 1;
            $subscriptionPurchase->save();

            $user = $deposit->user;
            $user->limit_contact = $subscriptionPurchase->number_of_contacts;
            $user->limit_emails = $subscriptionPurchase->number_of_emails;
            $user->plan_id = $subscriptionPurchase->id;
            $user->save();

            if ($subscriptionPurchase->frequency == 'monthly') {
                $user->subs_expired_at = Carbon::now()->addDays(30);
                $subscriptionPurchase->subs_expired_at = Carbon::now()->addDays(30);
            } elseif ($subscriptionPurchase->frequency == 'yearly') {
                $user->subs_expired_at = Carbon::now()->addDays(365);
                $subscriptionPurchase->subs_expired_at = Carbon::now()->addDays(365);
            }

            $user->save();
            $subscriptionPurchase->save();

            $fund = $this->newFund($deposit);
            $this->newTransaction($fund, 'Subscription Via');

            $deposit->save();
            $params = [
                'amount' => getAmount($deposit->amount, $basicControl->fraction_number),
                'currency' => $basicControl->base_currency,
                'planName' => $subscriptionPurchase->plan_name ?? null,
                'transaction' => $deposit->utr,
            ];

            $action = [
                "link" => "#",
                "icon" => "fa fa-money-bill-alt text-white"
            ];

            $this->sendMailSms($deposit->receiver, 'USER_BUY_PLAN', $params);
            $this->userPushNotification($deposit->receiver, 'USER_BUY_PLAN', $params, $action);
            $firebaseAction = route('user.fund.index');
            $this->userFirebasePushNotification($deposit->receiver, 'USER_BUY_PLAN', $params, $firebaseAction);

            $params = [
                'user' => optional($deposit->receiver)->username ?? null,
                'amount' => getAmount($deposit->amount, $basicControl->fraction_number),
                'currency' => $basicControl->base_currency,
                'planName' => $subscriptionPurchase->plan_name ?? null,
                'transaction' => $deposit->utr,
            ];

            $action = [
                "link" => '#',
                "icon" => "fa fa-money-bill-alt text-white"
            ];

            $this->adminMail('USER_BUY_PLAN_ADMIN', $params);
            $this->adminPushNotification('USER_BUY_PLAN_ADMIN', $params, $action);
            $this->adminFirebasePushNotification('USER_BUY_PLAN_ADMIN', $params);

        }
        return true;
    }

    public function newFund($deposit)
    {
        $fund = new Fund();
        $fund->user_id = $deposit->user_id;
        $fund->gateway_id = $deposit->payment_method_id;
        $fund->gateway_currency = $deposit->payment_method_currency;
        $fund->amount = $deposit->amount;
        $fund->percentage_charge = $deposit->percentage;
        $fund->fixed_charge = $deposit->charge_fixed;
        $fund->charge = $deposit->charge;
        $fund->final_amount = $deposit->payable_amount;
        $fund->payable_amount_base_currency = $deposit->payable_amount_base_currency;
        $fund->status = 1;
        $fund->transaction = $deposit->trx_id;
        $fund->detail = $deposit->information;
        $fund->save();

        return $fund;
    }

    public function newTransaction($fund, $remark): void
    {
        $transaction = new Transaction();
        $transaction->user_id = $fund->user_id;
        $transaction->amount = $fund->amount;
        $transaction->charge = getAmount($fund->charge);
        $transaction->trx_type = '+';
        $transaction->trx_id = $fund->transaction;
        $transaction->remarks = $remark . ' ' . optional($fund->gateway)->name;
        $fund->transactional()->save($transaction);
    }

}
