<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Models\Deposit;
use App\Models\Fund;
use App\Models\Gateway;
use App\Models\Plan;
use App\Models\SubscriptionPurchase;
use App\Traits\PaymentValidationCheck;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Str;
use Stevebauman\Purify\Facades\Purify;

class SubscriptionController extends Controller
{
    use PaymentValidationCheck;

    public function __construct()
    {
        $this->theme = template();
    }

    public function planValidate($subscriptionPlan)
    {
        $user = auth()->user();
        if ($subscriptionPlan->number_of_contacts < $user->use_contacts) {
            return [
                'status' => 'error',
                'message' => 'Please note that downgrading your plan is not available from your generated contact list.'
            ];
        } elseif ($subscriptionPlan->number_of_emails < $user->use_emails) {
            return [
                'status' => 'error',
                'message' => 'Please note that downgrading your plan is not available from your sending email list.'
            ];
        }

        return [
            'status' => 'success'
        ];
    }

    public function subscriptionPurchase(Request $request)
    {
        $subscriptionPlan = Plan::where('status', 1)->where('is_free', 0)->findOrFail($request->id);
        $isPlanValidate = $this->planValidate($subscriptionPlan);
        if ($isPlanValidate['status'] == 'error') {
            return back()->with('error', $isPlanValidate['message']);
        }

        if ($request->isMethod('get')) {
            $methods = Gateway::orderBy('sort_by', 'ASC')->where('status', 1)->get();
            return view($this->theme . 'user.subscription.create', compact('methods', 'subscriptionPlan'));
        } elseif ($request->isMethod('post')) {
            $purifiedData = Purify::clean($request->all());

            $validationRules = [
                'methodId' => 'required|integer|min:1|not_in:0',
                'supported_currency' => 'required',
            ];

            $validate = Validator::make($purifiedData, $validationRules);
            if ($validate->fails()) {
                return back()->withErrors($validate)->withInput();
            }

            $purifiedData = (object)$purifiedData;

            $amount = $subscriptionPlan->price;
            $methodId = $purifiedData->methodId;
            $currency = $purifiedData->supported_currency;

            $checkAmountValidate = $this->validationCheck($amount, $methodId, $currency);

            if ($checkAmountValidate['status'] == 'error') {
                return back()->with('error', $checkAmountValidate['msg']);
            }

            $method = Gateway::where('status', 1)->findOrFail($methodId);

            $fillData = new SubscriptionPurchase($subscriptionPlan->attributesToArray());
            $fillData['user_id'] = auth()->id();
            $fillData['plan_id'] = $subscriptionPlan->id;
            $fillData['currency'] = $currency;
            $fillData['payment_method_id'] = $method->id;
            $fillData->save();

            $deposit = Deposit::create([
                'user_id' => Auth::user()->id,
                'payment_method_id' => $checkAmountValidate['data']['gateway_id'],
                'payment_method_currency' => $checkAmountValidate['data']['currency'],
                'amount' => $amount,
                'percentage' => $checkAmountValidate['data']['percentage'],
                'charge_percentage' => $checkAmountValidate['data']['percentage_charge'],
                'charge_fixed' => $checkAmountValidate['data']['fixed_charge'],
                'charge' => $checkAmountValidate['data']['charge'],
                'payable_amount' => $checkAmountValidate['data']['payable_amount'],
                'payable_amount_base_currency' => $checkAmountValidate['data']['payable_amount_baseCurrency'],
                'trx_id' => strRandom(),
                'status' => 0,
                'depositable_type' => SubscriptionPurchase::class,
                'depositable_id' => $fillData->id,
            ]);

            if ($method->subscription_on) {
                return redirect()->route('user.subscription.process', $deposit->trx_id);
            } else {
                return redirect(route('payment.process', $deposit->trx_id));
            }
        }
    }

    public function subsConfirm(Request $request, $trx_id)
    {
        $deposit = Deposit::with('user', 'depositable')->where(['trx_id' => $trx_id, 'status' => 0])->first();
        $gateway = Gateway::findOrFail($deposit->payment_method_id);

        try {
            if ($gateway->code == 'square') {
                $getwayObj = 'App\\Services\\Subscription\\' . $gateway->code . '\\Payment';
                $data = $getwayObj::createSubscription($deposit, $gateway, $request);

                if ($data['status'] == 'success') {
                    if (isset($data['redirect_url'])) {
                        return redirect()->away($data['redirect_url']);
                    }
                    return redirect()->route('success');
                } else {
                    return back()->with('error', 'Invalid Payment');
                }
            }
        } catch (\Exception $exception) {
            return back()->with('error', $exception->getMessage());
        }

        if ($request->method() == "GET") {
            return view($this->theme . 'user.subscription.payment-method.' . $gateway->code, compact('gateway', 'deposit'));
        } elseif ($request->method() == 'POST') {

            try {
                $getwayObj = 'App\\Services\\Subscription\\' . $gateway->code . '\\Payment';
                $data = $getwayObj::createSubscription($deposit, $gateway, $request);

                if ($data['status'] == 'success') {
                    if (isset($data['redirect_url'])) {
                        return redirect()->away($data['redirect_url']);
                    }
                    return redirect()->route('success');
                } else {
                    return back()->with('error', 'Invalid Payment');
                }
            } catch (\Exception $e) {
                return back()->with('error', $e->getMessage());
            }
        }
    }

    public function subscriptionIpn(Request $request, $code, $trx_id = null)
    {

        try {
            $gateway = Gateway::where('code', $code)->first();
            if (!$gateway) throw new \Exception('Invalid Payment Gateway.');

            if (isset($trx_id)) {
                $deposit = Deposit::with('user')->where('trx_id', $trx_id)->first();
                if (!$deposit) throw new \Exception('Invalid Payment Request.');
            }
            $getwayObj = 'App\\Services\\Subscription\\' . $code . '\\Payment';
            $data = $getwayObj::ipn($request, $gateway, @$deposit, @$trx_id);

        } catch (\Exception $exception) {
            return back()->with('alert', $exception->getMessage());
        }
    }

    public function subsList()
    {
        $data['subscriptions'] = SubscriptionPurchase::own()->with(['gateway'])
            ->where('payment_status', 1)
            ->orderBy('id', 'desc')
            ->paginate(config('basic.paginate'));
        return view($this->theme . 'user.subscription.index', $data);
    }

    public function subsCancel($id)
    {
        $subscriptionPurchase = SubscriptionPurchase::own()->findOrFail($id);

        try {
            $code = $subscriptionPurchase->gateway->code;
            $getwayObj = 'App\\Services\\Subscription\\' . $code . '\\Payment';
            $data = $getwayObj::cancelSubscription($subscriptionPurchase);
            if ($data['status'] == 'success') {
                $subscriptionPurchase->status = 0;
                $subscriptionPurchase->cancel_at = Carbon::now();
                $subscriptionPurchase->save();
                return back()->with('success', 'Your subscription has been canceled');
            } else {
                return back()->with('alert', 'You can not cancel subscription');
            }
        } catch (\Exception $e) {
            return back()->with('alert', 'Something went wrong. Please try again');
        }
    }

    public function subscriptionPurchaseFree(Request $request)
    {
        $freePlan = Plan::where('status', 1)->where('is_free', 1)->findOrFail($request->id);

        $isPlanValidate = $this->planValidate($freePlan);
        if ($isPlanValidate['status'] == 'error') {
            return back()->with('error', $isPlanValidate['message']);
        }

        $user = auth()->user();

        DB::beginTransaction();
        try {
            $fillData = new SubscriptionPurchase($freePlan->attributesToArray());
            $fillData['user_id'] = $user->id;
            $fillData['plan_id'] = $freePlan->id;
            $fillData['payment_status'] = 1;
            $fillData['created_at'] = Carbon::now();
            $fillData['subs_expired_at'] = Carbon::now()->addDays(30);
            $fillData->save();

            $user->limit_contact = $freePlan->number_of_contacts;
            $user->limit_emails = $freePlan->number_of_emails;
            $user->plan_id = $fillData->id;
            $user->subs_expired_at = Carbon::now()->addDays(30);
            $user->save();

            DB::commit();
            return redirect()->route('user.dashboard')->with('success', 'Purchased Successfull');
        } catch (\Exception $exception) {
            DB::rollBack();
            return back()->with('error', $exception->getMessage());
        }
    }

    public function subsRenew($purchaseSubsId)
    {
        $subsPurchase = SubscriptionPurchase::findOrFail($purchaseSubsId);
        if (!$subsPurchase->payment_method_id) {
            $subsPurchase->subs_expired_at = Carbon::now()->addDays(30);
            $subsPurchase->save();

            $user = $subsPurchase->user;
            $user->subs_expired_at = Carbon::now()->addDays(30);
            $user->save();

            return back()->with('success', 'Renew Successfully');
        }

        $checkAmountValidate = $this->validationCheck($subsPurchase->price, $subsPurchase->payment_method_id, $subsPurchase->currency);

        if ($checkAmountValidate['status'] == 'error') {
            return back()->with('error', $checkAmountValidate['msg']);
        }

        $deposit = Deposit::create([
            'user_id' => $subsPurchase->user_id,
            'payment_method_id' => $checkAmountValidate['data']['gateway_id'],
            'payment_method_currency' => $checkAmountValidate['data']['currency'],
            'amount' => $subsPurchase->price,
            'percentage' => $checkAmountValidate['data']['percentage'],
            'charge_percentage' => $checkAmountValidate['data']['percentage_charge'],
            'charge_fixed' => $checkAmountValidate['data']['fixed_charge'],
            'charge' => $checkAmountValidate['data']['charge'],
            'payable_amount' => $checkAmountValidate['data']['payable_amount'],
            'payable_amount_base_currency' => $checkAmountValidate['data']['payable_amount_baseCurrency'],
            'trx_id' => strRandom(),
            'status' => 0,
            'depositable_type' => SubscriptionPurchase::class,
            'depositable_id' => $subsPurchase->id,
        ]);

        return redirect(route('payment.process', $deposit->trx_id));
    }
}
