<?php

namespace App\Http\Controllers\User;

use App\Exports\ExportSingleSendActivityEmail;
use App\Http\Controllers\Controller;
use App\Http\Requests\SingleSendSaveRequest;
use App\Jobs\SingleSendEmails;
use App\Models\ContactType;
use App\Models\CustomField;
use App\Models\EmailTemplate;
use App\Models\Segment;
use App\Models\Sender;
use App\Models\SingleSend;
use App\Models\SingleSendActivity;
use App\Models\UnsubscribeGroup;
use App\Traits\Activity;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Maatwebsite\Excel\Facades\Excel;
use Ramsey\Uuid\Uuid;
use DateTime;


class SingleSendController extends Controller
{
    use Activity;

    public function __construct()
    {
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            $this->user = auth()->user();
            return $next($request);
        });
        $this->theme = template();
    }

    public function index(Request $request)
    {
        $search = $request->all();
        $dateSearch = $request->send_at;
        $date = preg_match("/^[0-9]{2,4}\-[0-9]{1,2}\-[0-9]{1,2}$/", $dateSearch);

        $data['singleSends'] = SingleSend::own()->orderBy('id', 'desc')
            ->when(isset($search['single_send_name']), function ($query) use ($search) {
                return $query->where('single_send_name', 'LIKE', '%' . $search['single_send_name'] . '%');
            })
            ->when($date == 1, function ($query) use ($dateSearch) {
                return $query->whereDate("sent_at", $dateSearch);
            })
            ->when(isset($search['status']) && $search['status'] != null, function ($query) use ($search) {
                if ($search['status'] == 'sent') {
                    return $query->where('status', 3);
                } elseif ($search['status'] == 'schedule') {
                    return $query->where('schedule_type', 1)->where('status', 1);
                } elseif ($search['status'] == 'running') {
                    return $query->where('status', 2);
                }
            })
            ->paginate(basicControl()->paginate);
        return view($this->theme . 'user.single_send.index', $data);
    }

    public function showTemplate()
    {
        $data['templates'] = EmailTemplate::select(['id', 'user_id', 'template_name', 'preview_image', 'driver', 'custom'])
            ->where('user_id', null)->orWhere('user_id', auth()->id())->get();
        return view($this->theme . 'user.single_send.show_template', $data);
    }

    public function singleSendCreate($templateId)
    {
        $template = EmailTemplate::findOrFail($templateId);
        $singleSend = new SingleSend();
        $singleSend->user_id = $this->user->id;
        $singleSend->template_id = $template->id;
        $singleSend->status = 0;
        $singleSend->utr = (Uuid::uuid1())->toString();
        $singleSend->template_json = $template->json_code;
        $singleSend->template_html = $template->html_code;
        $singleSend->save();

        return redirect()->route('user.singleSendSetting', $singleSend->utr);
    }

    public function singleSendSetting($utr)
    {
        $data['singleSend'] = SingleSend::own()->whereIn('status', ['0', '1', '2'])->where('utr', $utr)->firstOrFail();
        $data['senders'] = Sender::own()->where('is_verified', 1)->orderBy('id', 'desc')->get();
        $data['contactTypes'] = ContactType::own(1)->withCount(['contacts'])->get();
        $data['segments'] = Segment::own()->withCount(['segmentEmails'])->orderBy('id', 'desc')->get();
        $data['unsubscribeGroups'] = UnsubscribeGroup::own()->withCount(['UnsubscribeClients'])->orderBy('id', 'desc')->get();
        $data['fields'] = CustomField::own()->orwhere('is_reserved', 1)->get();
        $data['isClassicEditor'] = ($data['singleSend']->template_id == 1) ? false : ($data['singleSend']->template_json === "{}");
        return view($this->theme . 'user.single_send.editor', $data);
    }

    public function singleSendDesignSave(Request $request, $utr)
    {
        $singleSend = SingleSend::own()->whereIn('status', ['0', '1', '2'])->where('utr', $utr)->first();
        $reqData = json_decode($request->getContent(), true);
        if ($singleSend) {
            $singleSend->template_json = $reqData['json_design'] ?? '{}';
            $singleSend->template_html = $reqData['html_design'];
            $singleSend->save();
        }
        return response()->json(['status' => 'success']);
    }

    public function singleSendSave(SingleSendSaveRequest $request, $utr)
    {
        $singleSend = SingleSend::own()->whereIn('status', ['0', '1', '2'])->where('utr', $utr)->first();
        $sender = Sender::own()->select(['id', 'is_verified', 'from_email', 'reply_to_email', 'from_name', 'company_address'])->where('is_verified', 1)->find($request->sender_id);

        if ($singleSend && $sender) {
            $singleSend->sender_id = $sender->id;
            $singleSend->unsubscribe_group_id = $request->unsubscribe_group_id;
            $singleSend->single_send_name = $request->single_send_name;
            $singleSend->from_sender = $sender->from_email;
            $singleSend->reply_to = $sender->reply_to_email;
            $singleSend->sender_name = $sender->from_name;
            $singleSend->sender_address = $sender->company_address;
            $singleSend->subject = $request->subject;
            $singleSend->preheader = $request->preheader;
            $singleSend->schedule_type = $request->schedule_type;
            $singleSend->schedule_date = $request->schedule_date;
            $singleSend->schedule_time = $request->schedule_time;
            $singleSend->status = 1;

            $recipients = [];
            if ($request->recipient_id) {
                foreach ($request->recipient_id as $recipent) {
                    $getData = getIdAndTable($recipent);
                    $recipients[] = [
                        'id' => $getData['id'],
                        'table' => $getData['table'],
                    ];
                }
            }

            $excludeRecipients = [];
            if ($request->exclude_specific_recipient_id) {
                foreach ($request->exclude_specific_recipient_id as $excludeRecipent) {
                    $getData = getIdAndTable($excludeRecipent);
                    $excludeRecipients[] = [
                        'id' => $getData['id'],
                        'table' => $getData['table'],
                    ];
                }
            }

            $singleSend->recipient_id = $recipients;
            $singleSend->exclude_specific_recipient_id = $excludeRecipients;
            $singleSend->save();

            if ($singleSend->schedule_type == '0') {
                $singleSend->status = 2;
                $singleSend->sent_at = Carbon::now();
                $singleSend->save();

                $basicControl = basicControl();
                $params = ['emailThrottleTime' => $basicControl->emailThrottleTime, 'emailThrottleStatus' => $basicControl->emailThrottleStatus];
                dispatch(new SingleSendEmails($singleSend, url('/'), $params));
            }

            $message = 'We have finished processing your request to create single send';
            $this->emailNotification($message, route('user.singleSendList'), 'View your single send');

            $route = route('user.singleSendList');
            $this->userActivity(
                "You have finished processing to initialize <a href='$route' class='text-primary'>$singleSend->single_send_name</a> single send!"
            );

            session()->flash('success', 'Initialize Successfully');
            return response()->json(['status' => 'success', 'route' => route('user.singleSendList')]);
        }
    }

    public function singleSendPreview($utr)
    {
        $data['singleSend'] = SingleSend::own()->where('utr', $utr)->firstOrFail();
        return view($this->theme . 'user.single_send.preview', $data);
    }

    public function singleSendDuplicate($utr)
    {
        $singleSend = SingleSend::own()->where('utr', $utr)->firstOrFail();
        $duplicateItem = new SingleSend();
        $duplicateItem->user_id = $singleSend->user_id;
        $duplicateItem->template_id = $singleSend->template_id;
        $duplicateItem->recipient_id = $singleSend->recipient_id;
        $duplicateItem->exclude_specific_recipient_id = $singleSend->exclude_specific_recipient_id;
        $duplicateItem->sender_id = $singleSend->sender_id;
        $duplicateItem->unsubscribe_group_id = $singleSend->unsubscribe_group_id;
        $duplicateItem->single_send_name = $singleSend->single_send_name;
        $duplicateItem->from_sender = $singleSend->from_sender;
        $duplicateItem->reply_to = $singleSend->reply_to;
        $duplicateItem->sender_name = $singleSend->sender_name;
        $duplicateItem->sender_address = $singleSend->sender_address;
        $duplicateItem->subject = $singleSend->subject;
        $duplicateItem->preheader = $singleSend->preheader;
        $duplicateItem->schedule_type = $singleSend->schedule_type;
        $duplicateItem->schedule_date = $singleSend->schedule_date;
        $duplicateItem->schedule_time = $singleSend->schedule_time;
        $duplicateItem->status = 0;
        $duplicateItem->utr = (Uuid::uuid1())->toString();
        $duplicateItem->template_json = $singleSend->template_json;
        $duplicateItem->template_html = $singleSend->template_html;
        $duplicateItem->save();

        $message = 'We have finished processing your request to duplicate single send';
        $this->emailNotification($message, route('user.singleSendList'), 'View your single send');

        $route = route('user.singleSendList');
        $this->userActivity(
            "You have finished processing to duplicate <a href='$route' class='text-primary'>$duplicateItem->single_send_name</a> single send!"
        );

        return redirect()->route('user.singleSendSetting', $duplicateItem->utr);
    }

    public function singleSendDelete($utr)
    {
        $singleSend = SingleSend::own()->where('status', '!=', 2)->where('utr', $utr)->firstOrFail();
        $singleSend->delete();
        $message = 'We have finished processing your request to delete single send';
        $this->emailNotification($message, route('user.singleSendList'), 'View your single send');

        $route = route('user.singleSendList');
        $this->userActivity(
            "You have finished processing to delete <a href='$route' class='text-primary'>$singleSend->single_send_name</a> single send!"
        );

        return back()->with('success', 'Deleted Successfully');
    }

    public function singleSendBulkDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select ID.');
            return response()->json(['error' => 1]);
        } else {
            SingleSend::own()->whereIn('id', $request->strIds)->where('status', '!=', 2)->get()->map(function ($query) {
                $query->delete();
            });
            $message = 'We have finished processing your request to delete multiple single send';
            $this->emailNotification($message, route('user.singleSendList'), 'View your single send');

            $this->userActivity(
                "You have finished processing to multiple delete single send!"
            );

            session()->flash('success', 'Deleted Successfully');
            return response()->json(['status' => 'success']);
        }
    }

    public function singleSendStats($utr)
    {
        $singleSend = SingleSend::own()->with(['sender', 'template', 'unsubscribeGroup', 'singleSendActivities'])
            ->withCount(['singleSendActivitiesUnsubscribes'])->where('utr', $utr)->firstOrFail();
        $singleSend->unsubscribes = $singleSend->single_send_activities_unsubscribes_count;
        $singleSend->save();


        $data['deliveredPercent'] = $singleSend->getPercent()['deliveredPercent'];
        $data['bouncesPercent'] = $singleSend->getPercent()['bouncesPercent'];
        $data['uniqueOpenPercent'] = $singleSend->getPercent()['uniqueOpenPercent'];
        $data['unsubscribesPercent'] = $singleSend->getPercent()['unsubscribesPercent'];

        $data['contactTypes'] = ContactType::own(1)->get();
        $data['segments'] = Segment::own()->get();


        $data['horizontalBarChat'] = [$singleSend->triggered, $singleSend->delivered, $singleSend->bounces,
            $singleSend->unique_opens, $singleSend->total_opens, $singleSend->unsubscribes];

        $data['lineChartUniqueOpen'] = DB::table('single_send_activities')
            ->select(DB::raw('triggered_date AS x'), DB::raw('COUNT(*) AS y'))
            ->where('single_send_id', $singleSend->id)
            ->where('type', '3')
            ->where('user_id', $this->user->id)
            ->groupBy('triggered_date')
            ->get()
            ->toArray();


        $start = Carbon::now()->firstOfMonth();
        $end = Carbon::now()->lastOfMonth();

        $transactions = SingleSendActivity::select(DB::raw("DATE_FORMAT(triggered_date, '%j') as day"))
            ->whereBetween('triggered_date', [$start, $end])
            ->selectRaw("COUNT(CASE WHEN type = '3' THEN 1 END) as `Unique`")
            ->selectRaw("COUNT(CASE WHEN type = '4' THEN 1 END) as `Unsubscribe`")
            ->selectRaw("triggered_date as `triggered_date`")
            ->where('user_id', $this->user->id)
            ->where('single_send_id', $singleSend->id)
            ->groupBy('day')
            ->get();

        $labels = [];
        $dataUnique = [];
        $dataUnsubscribe = [];
        $start = new DateTime($start);
        $end = new DateTime($end);

        for ($day = $start; $day <= $end; $day->modify('+1 day')) {
            $i = $day->format('j');
            $labels[] = $day->format('jS M');
            $currentUnique = 0;
            $currentUnsubscribe = 0;

            if (isset($transactions)) {
                foreach ($transactions as $key => $transaction) {
                    if (Carbon::parse($transaction->triggered_date) == $day) {
                        $currentUnique = $transaction->Unique;
                        $currentUnsubscribe = $transaction->Unsubscribe;
                        break;
                    }
                }
            }

            $dataUnique[] = $currentUnique;
            $dataUnsubscribe[] = $currentUnsubscribe;
        }

        $data['labels'] = $labels;
        $data['dataUnique'] = $dataUnique;
        $data['dataUnsubscribe'] = $dataUnsubscribe;
        $data['singleSend'] = $singleSend;
        return view($this->theme . 'user.single_send.stats', $data);
    }

    public function singleSendGetChart(Request $request, $utr)
    {
        $singleSend = SingleSend::own()->select(['id', 'utr'])->where('utr', $utr)->first();
        $start = $request->start;
        $end = $request->end;

        $transactions = SingleSendActivity::select(DB::raw("DATE_FORMAT(triggered_date, '%j') as day"))
            ->whereBetween('triggered_date', [$start, $end])
            ->selectRaw("COUNT(CASE WHEN type = '3' THEN 1 END) as `Unique`")
            ->selectRaw("COUNT(CASE WHEN type = '4' THEN 1 END) as `Unsubscribe`")
            ->selectRaw("triggered_date as `triggered_date`")
            ->where('user_id', $this->user->id)
            ->where('single_send_id', $singleSend->id)
            ->groupBy('day')
            ->get();

        $labels = [];
        $dataUnique = [];
        $dataUnsubscribe = [];
        $start = new DateTime($start);
        $end = new DateTime($end);

        for ($day = $start; $day <= $end; $day->modify('+1 day')) {
            $i = $day->format('j');
            $labels[] = $day->format('jS M');
            $currentUnique = 0;
            $currentUnsubscribe = 0;

            if (isset($transactions)) {
                foreach ($transactions as $key => $transaction) {
                    if (Carbon::parse($transaction->triggered_date) == $day) {
                        $currentUnique = $transaction->Unique;
                        $currentUnsubscribe = $transaction->Unsubscribe;
                        break;
                    }
                }
            }

            $dataUnique[] = $currentUnique;
            $dataUnsubscribe[] = $currentUnsubscribe;
        }

        $data['labels'] = $labels;
        $data['dataUnique'] = $dataUnique;
        $data['dataUnsubscribe'] = $dataUnsubscribe;

        return response()->json($data);
    }

    public function singleSendLogs(Request $request, $utr)
    {
        $search = $request->all();
        $dateSearch = $request->created_at;
        $date = preg_match("/^[0-9]{2,4}\-[0-9]{1,2}\-[0-9]{1,2}$/", $dateSearch);

        if ($request->logs == 'delivered') {
            $data['title'] = 'Delivered';
            $type = 1;
        } elseif ($request->logs == 'bounces') {
            $data['title'] = 'Bounces';
            $type = 2;
        } elseif ($request->logs == 'unique_opens') {
            $data['title'] = 'Unique Opens';
            $type = 3;
        } elseif ($request->logs == 'unsubscribe') {
            $data['title'] = 'Unsubscribe';
            $type = 4;
        } else {
            abort('404');
        }

        $singleSend = SingleSend::own()->select(['id', 'utr', 'single_send_name'])->latest()->where('utr', $utr)->firstOrFail();
        $data['singleSendRecipients'] = SingleSendActivity::own()->where('single_send_id', $singleSend->id)
            ->where('type', $type)
            ->when(isset($search['email']), function ($query) use ($search) {
                return $query->where('email', 'LIKE', '%' . $search['email'] . '%');
            })
            ->when($date == 1, function ($query) use ($dateSearch) {
                return $query->whereDate("created_at", $dateSearch);
            })
            ->orderBy('id', 'desc')
            ->paginate(basicControl()->paginate);

        $data['logsType'] = $request->logs;
        $data['type'] = $type;
        return view($this->theme . 'user.single_send.log.index', $data, compact('singleSend'));
    }

    public function recipientExportCsv($singleSendId, $type)
    {
        SingleSend::own()->select(['id'])->findOrFail($singleSendId);
        $this->userActivity(
            "You have finished processing to export single send recipients!"
        );
        return Excel::download(new ExportSingleSendActivityEmail($singleSendId, $type), 'emails.csv');
    }

}
