<?php

namespace App\Http\Controllers\User;

use App\Http\Controllers\Controller;
use App\Http\Requests\NotificationRequest;
use App\Models\Notification;
use App\Traits\Activity;
use Illuminate\Http\Request;

class NotificationController extends Controller
{
    use Activity;

    public function __construct()
    {
        $this->middleware(['auth']);
        $this->middleware(function ($request, $next) {
            $this->user = auth()->user();
            return $next($request);
        });
        $this->theme = template();
    }

    public function index(Request $request)
    {
        $search = $request->all();
        $data['notifications'] = Notification::own()->orderBy('id', 'desc')
            ->when(isset($search['email']), function ($query) use ($search) {
                $query->where('email_address', 'LIKE', '%' . $search['email'] . '%');
            })
            ->when(isset($search['owner']), function ($query) use ($search) {
                $query->where('email_owner', 'LIKE', '%' . $search['owner'] . '%');
            })
            ->paginate(basicControl()->paginate);
        return view($this->theme . 'user.email_notification.index', $data);
    }

    public function emailNotificationSave(NotificationRequest $request)
    {
        $notification = new Notification();
        $fillData = $request->only($notification->getFillable());
        $fillData['user_id'] = $this->user->id;
        $notification->fill($fillData)->save();

        $route = route('user.emailNotification');
        $this->userActivity(
            "You have finished processing to create <a href='$route' class='text-primary'>notification</a>!"
        );

        session()->flash('success', 'Added Successfully');
        return response()->json(['status' => 'success']);
    }

    public function emailNotificationDelete($id)
    {
        $notification = Notification::own()->findOrFail($id);
        $notification->delete();

        $route = route('user.emailNotification');
        $this->userActivity(
            "You have finished processing to delete <a href='$route' class='text-primary'>notification</a>!"
        );

        return back()->with('success', 'Deleted Successfully');
    }

    public function emailNotificationBulkDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select ID.');
            return response()->json(['error' => 1]);
        } else {
            Notification::own()->whereIn('id', $request->strIds)->get()->map(function ($query) {
                $query->delete();
            });

            $route = route('user.emailNotification');
            $this->userActivity(
                "You have finished processing to multiple delete <a href='$route' class='text-primary'>notification</a>!"
            );

            session()->flash('success', 'Deleted Successfully');
            return response()->json(['status' => 'success']);
        }
    }
}
