<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\UnsubscribesGroupRequest;
use App\Http\Requests\UnsubscribesStore;
use App\Models\UnsubscribeClient;
use App\Models\UnsubscribeGroup;
use App\Models\User;
use App\Traits\Activity;
use App\Traits\ApiValidation;
use Carbon\Carbon;
use Illuminate\Http\Request;

class UnsubscribeController extends Controller
{
    use ApiValidation, Activity;

    public function globalUnsubscribes(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['unSubscribers'] = UnsubscribeClient::where('user_id', $user->id)->orderBy('id', 'desc')
                ->get();

            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Contact with your provider'));
    }

    public function globalUnsubscribesSave(UnsubscribesStore $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $emailString = $request->email_address;
            $emailArray = explode(',', $emailString);
            foreach ($emailArray as $email) {
                $unsubscribeClients = new UnsubscribeClient();
                $fillData = $request->only($unsubscribeClients->getFillable());
                $fillData['user_id'] = $user->id;
                $fillData['email_address'] = $email;
                $unsubscribeClients->fill($fillData)->save();
            }
            $this->userActivity(
                "A new global unsubscribers was created!", $user->id, $request->ip()
            );
            return response()->json($this->withSuccess('Added Successfully'));
        }
        return response()->json($this->withErrors('Contact with your provider'));
    }

    public function globalUnsubscribesDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $unsubscribes = UnsubscribeClient::where('user_id', $user->id)->find($request->id);
            if ($unsubscribes) {
                $unsubscribes->delete();
                $message = 'We have finished processing your request to delete unsubscribers';
                $this->emailNotification($message, route('user.globalUnsubscribes'), 'View your unsubscribers', $user->id);

                $this->userActivity(
                    "A single global unsubscribers was deleted!", $user->id, $request->ip()
                );

                return response()->json($this->withSuccess('Deleted Successfully'));
            }
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function globalUnsubscribesBulkDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            if ($request->strIds == null) {
                return response()->json($this->withErrors('No Id Found.'));
            } else {
                UnsubscribeClient::where('user_id', $user->id)->whereIn('id', $request->strIds)->get()->map(function ($query) {
                    $query->delete();
                });
                $message = 'We have finished processing your request to delete multiple unsubscribers';
                $this->emailNotification($message, route('user.globalUnsubscribes'), 'View your unsubscribers', $user->id);

                $this->userActivity(
                    "Multiple  global unsubscribers was deleted!", $user->id, $request->ip()
                );

                return response()->json($this->withSuccess('Deleted Successfully'));
            }
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function unsubscribesGroup(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['unsubscribesGroups'] = UnsubscribeGroup::where('user_id', $user->id)
                ->withCount('UnsubscribeClients')->orderBy('id', 'desc')->get();
            return response()->json($this->withSuccess($data));
        }

        return response()->json($this->withErrors('Record Not Found'));
    }

    public function unsubscribesGroupSave(UnsubscribesGroupRequest $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $unsubscribeGroup = new UnsubscribeGroup();
            $fillData = $request->only($unsubscribeGroup->getFillable());
            $fillData['user_id'] = $user->id;
            $fillData['rand_id'] = rand(00000, 99999);
            $unsubscribeGroup->fill($fillData)->save();

            $route = route('user.unsubscribesGroup');
            $this->userActivity(
                "you have finished processing to create <a href='$route' class='text-primary'>$unsubscribeGroup->group_name</a> unsubscribe group!",
                $user->id, $request->ip()
            );

            return response()->json($this->withSuccess('Saved Successfully'));
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function unsubscribesGroupEdit(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['unsubscribesGroup'] = UnsubscribeGroup::where('user_id', $user->id)->find($request->id);
            if ($data['unsubscribesGroup']) {
                return response()->json($this->withSuccess($data));
            }
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function unsubscribesGroupUpdate(UnsubscribesGroupRequest $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $unsubscribeGroup = UnsubscribeGroup::where('user_id', $user->id)->find($request->id);
            if ($unsubscribeGroup) {
                $fillData = $request->only($unsubscribeGroup->getFillable());
                $unsubscribeGroup->fill($fillData)->save();

                $route = route('user.unsubscribesGroup');
                $this->userActivity(
                    "you have finished processing to update <a href='$route' class='text-primary'>$unsubscribeGroup->group_name</a> unsubscribe group!",
                    $user->id, $request->ip()
                );

                return response()->json($this->withSuccess('Updated Successfully'));
            }
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function unsubscribesGroupManuallyAdd(UnsubscribesStore $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $group = UnsubscribeGroup::where('user_id', $user->id)->find($request->unsubscribe_groups_id);
            if ($group) {
                $emailString = $request->email_address;
                $emailArray = explode(',', $emailString);
                foreach ($emailArray as $email) {
                    $unsubscribeClients = new UnsubscribeClient();
                    $fillData = $request->only($unsubscribeClients->getFillable());
                    $fillData['user_id'] = $user->id;
                    $fillData['unsubscribe_groups_id'] = $group->id;
                    $fillData['email_address'] = $email;
                    $unsubscribeClients->fill($fillData)->save();
                }

                $this->userActivity(
                    "Unsubscribe group recipients was added!", $user->id, $request->ip()
                );
                return response()->json($this->withSuccess('Added Successfully'));
            }
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function unsubscribesRecipients(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $group = UnsubscribeGroup::where('user_id', $user->id)->find($request->groupId);
            if ($group) {
                $data['recipients'] = UnsubscribeClient::where('user_id', $user->id)->where('unsubscribe_groups_id', $group->id)
                    ->orderBy('id', 'desc')
                    ->get();
                return response()->json($this->withSuccess($data));
            }
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function unsubscribesRecipientDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $recipient = UnsubscribeClient::where('user_id', $user->id)->find($request->id);
            if ($recipient) {
                $recipient->delete();
                $message = 'We have finished processing your request to delete unsubscribes recipients';
                $this->emailNotification($message, route('user.unsubscribesRecipients', $recipient->unsubscribe_groups_id), 'View your recipients', $user->id);

                $route = route('user.unsubscribesRecipients', $recipient->unsubscribe_groups_id);
                $this->userActivity(
                    "you have finished processing to delete <a href='$route' class='text-primary'>unsubscribes recipients</a>!",
                    $user->id, $request->ip()
                );

                return response()->json($this->withSuccess('Deleted Successfully'));
            }
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function unsubscribesRecipientBulkDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            if ($request->strIds == null) {
                return response()->json($this->withErrors('Id not found'));
            } else {
                UnsubscribeClient::where('user_id', $user->id)->whereIn('id', $request->strIds)->get()->map(function ($query) {
                    $query->delete();
                });
                $message = 'We have finished processing your request to delete multiple unsubscribes recipients';
                $this->emailNotification($message, null, null, $user->id);
                $this->userActivity("you have finished processing to unsubscribe group recipients multiple delete!", $user->id, $request->ip());
                return response()->json($this->withSuccess('Deleted Successfully'));
            }
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

}
