<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\SingleSendSaveRequest;
use App\Jobs\SingleSendEmails;
use App\Models\CustomField;
use App\Models\EmailTemplate;
use App\Models\Sender;
use App\Models\SingleSend;
use App\Traits\Activity;
use App\Traits\ApiValidation;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Ramsey\Uuid\Uuid;

class SingleSendController extends Controller
{
    use ApiValidation, Activity;

    public function index(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['singleSends'] = SingleSend::where('user_id', $user->id)->orderBy('id', 'desc')
                ->get();
            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function singleSendCreate(SingleSendSaveRequest $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $sender = Sender::where('user_id', $user->id)->select(['id', 'is_verified', 'from_email', 'reply_to_email'])
                ->where('is_verified', 1)->find($request->sender_id);

            $template = EmailTemplate::find($request->template_id);
            if (!$template) {
                return response()->json($this->withErrors('Template not found'));
            }

            if ($sender && $template) {
                $singleSend = new SingleSend();

                $singleSend->user_id = $user->id;
                $singleSend->template_id = $template->id;
                $singleSend->utr = (Uuid::uuid1())->toString();
                $singleSend->template_json = $request->template_json ?? $template->json_code;
                $singleSend->template_html = $request->template_html ?? $template->html_code;
                if ($request->template_html) {
                    $singleSend->template_json = "{}";
                }

                $singleSend->sender_id = $sender->id;
                $singleSend->unsubscribe_group_id = $request->unsubscribe_group_id;
                $singleSend->single_send_name = $request->single_send_name;
                $singleSend->from_sender = $sender->from_email;
                $singleSend->reply_to = $sender->reply_to_email;
                $singleSend->sender_name = $sender->from_name;
                $singleSend->sender_address = $sender->company_address;
                $singleSend->subject = $request->subject;

                $singleSend->preheader = $request->preheader;
                $singleSend->schedule_type = $request->schedule_type;
                $singleSend->schedule_date = $request->schedule_date;
                $singleSend->schedule_time = $request->schedule_time;
                $singleSend->status = 1;

                $recipients = [];
                if ($request->recipient_id) {
                    foreach ($request->recipient_id as $recipent) {
                        $getData = getIdAndTable($recipent);
                        $recipients[] = [
                            'id' => $getData['id'],
                            'table' => $getData['table'],
                        ];
                    }
                }

                $excludeRecipients = [];
                if ($request->exclude_specific_recipient_id) {
                    foreach ($request->exclude_specific_recipient_id as $excludeRecipent) {
                        $getData = getIdAndTable($excludeRecipent);
                        $excludeRecipients[] = [
                            'id' => $getData['id'],
                            'table' => $getData['table'],
                        ];
                    }
                }

                $singleSend->recipient_id = $recipients;
                $singleSend->exclude_specific_recipient_id = $excludeRecipients;
                $singleSend->save();

                if ($singleSend->schedule_type == '0') {
                    $singleSend->status = 2;
                    $singleSend->sent_at = Carbon::now();
                    $singleSend->save();

                    $basicControl = basicControl();
                    $params = ['emailThrottleTime' => $basicControl->emailThrottleTime, 'emailThrottleStatus' => $basicControl->emailThrottleStatus];
                    dispatch(new SingleSendEmails($singleSend, url('/'), $params));
                }

                $message = 'We have finished processing your request to create single send';
                $this->emailNotification($message, route('user.singleSendList'), 'View your single send', $user->id);

                $route = route('user.singleSendList');
                $this->userActivity(
                    "You have finished processing to initialize <a href='$route' class='text-primary'>$singleSend->single_send_name</a> single send!",
                    $user->id, $request->ip()
                );

                $data['utr'] = $singleSend->utr;
                return response()->json($this->withSuccess($data));
            }
        }
    }

    public function singleSendUpdate(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $sender = Sender::where('user_id', $user->id)->select(['id', 'is_verified', 'from_email', 'reply_to_email'])
                ->where('is_verified', 1)->find($request->sender_id);

            $template = EmailTemplate::find($request->template_id);
            if (!$template) {
                return response()->json($this->withErrors('Template not found'));
            }
            $singleSend = SingleSend::where('user_id', $user->id)->where('utr', $request->utr)
                ->whereIn('status', [0, 1])->first();

            if ($sender && $template && $singleSend) {
                $singleSend->template_id = $template->id;
                $singleSend->template_json = $request->template_json ?? $template->json_code;
                $singleSend->template_html = $request->template_html ?? $template->html_code;
                if ($request->template_html) {
                    $singleSend->template_json = "{}";
                }

                $singleSend->sender_id = $sender->id;
                $singleSend->unsubscribe_group_id = $request->unsubscribe_group_id;
                $singleSend->single_send_name = $request->single_send_name;
                $singleSend->from_sender = $sender->from_email;
                $singleSend->reply_to = $sender->reply_to_email;
                $singleSend->sender_name = $sender->from_name;
                $singleSend->sender_address = $sender->company_address;
                $singleSend->subject = $request->subject;

                $singleSend->preheader = $request->preheader;
                $singleSend->schedule_type = $request->schedule_type;
                $singleSend->schedule_date = $request->schedule_date;
                $singleSend->schedule_time = $request->schedule_time;
                $singleSend->status = 1;

                $recipients = [];
                if ($request->recipient_id) {
                    foreach ($request->recipient_id as $recipent) {
                        $getData = getIdAndTable($recipent);
                        $recipients[] = [
                            'id' => $getData['id'],
                            'table' => $getData['table'],
                        ];
                    }
                }

                $excludeRecipients = [];
                if ($request->exclude_specific_recipient_id) {
                    foreach ($request->exclude_specific_recipient_id as $excludeRecipent) {
                        $getData = getIdAndTable($excludeRecipent);
                        $excludeRecipients[] = [
                            'id' => $getData['id'],
                            'table' => $getData['table'],
                        ];
                    }
                }

                $singleSend->recipient_id = $recipients;
                $singleSend->exclude_specific_recipient_id = $excludeRecipients;
                $singleSend->save();

                if ($singleSend->schedule_type == '0') {
                    $singleSend->status = 2;
                    $singleSend->sent_at = Carbon::now();
                    $singleSend->save();

                    $basicControl = basicControl();
                    $params = ['emailThrottleTime' => $basicControl->emailThrottleTime, 'emailThrottleStatus' => $basicControl->emailThrottleStatus];
                    dispatch(new SingleSendEmails($singleSend, url('/'), $params));
                }

                $message = 'We have finished processing your request to create single send';
                $this->emailNotification($message, route('user.singleSendList'), 'View your single send', $user->id);

                $route = route('user.singleSendList');
                $this->userActivity(
                    "You have finished processing to update <a href='$route' class='text-primary'>$singleSend->single_send_name</a> single send!",
                    $user->id, $request->ip()
                );

                $data['utr'] = $singleSend->utr;
                return response()->json($this->withSuccess($data));
            }
        }
    }

    public function singleSendDuplicate(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $singleSend = SingleSend::where('user_id', $user->id)->where('utr', $request->utr)->first();
            if ($singleSend) {
                $duplicateItem = new SingleSend();
                $duplicateItem->user_id = $singleSend->user_id;
                $duplicateItem->template_id = $singleSend->template_id;
                $duplicateItem->recipient_id = $singleSend->recipient_id;
                $duplicateItem->exclude_specific_recipient_id = $singleSend->exclude_specific_recipient_id;
                $duplicateItem->sender_id = $singleSend->sender_id;
                $duplicateItem->unsubscribe_group_id = $singleSend->unsubscribe_group_id;
                $duplicateItem->single_send_name = $singleSend->single_send_name;
                $duplicateItem->from_sender = $singleSend->from_sender;
                $duplicateItem->reply_to = $singleSend->reply_to;
                $duplicateItem->sender_name = $singleSend->sender_name;
                $duplicateItem->sender_address = $singleSend->sender_address;
                $duplicateItem->subject = $singleSend->subject;
                $duplicateItem->preheader = $singleSend->preheader;
                $duplicateItem->schedule_type = $singleSend->schedule_type;
                $duplicateItem->schedule_date = $singleSend->schedule_date;
                $duplicateItem->schedule_time = $singleSend->schedule_time;
                $duplicateItem->status = 0;
                $duplicateItem->utr = (Uuid::uuid1())->toString();
                $duplicateItem->template_json = $singleSend->template_json;
                $duplicateItem->template_html = $singleSend->template_html;
                $duplicateItem->save();

                $message = 'We have finished processing your request to duplicate single send';
                $this->emailNotification($message, route('user.singleSendList'), 'View your single send', $user->id);

                $route = route('user.singleSendList');
                $this->userActivity(
                    "You have finished processing to duplicate <a href='$route' class='text-primary'>$duplicateItem->single_send_name</a> single send!",
                    $user->id, $request->ip()
                );

                return response()->json($this->withSuccess('Duplicated Successfully'));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function singleSendDeatils(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['singleSend'] = SingleSend::where('user_id', $user->id)->where('utr', $request->utr)->first();
            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function singleSendDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'));
        if ($user) {
            $singleSend = SingleSend::where('user_id', $user->id)->where('status', '!=', 2)
                ->where('utr', $request->utr)->first();
            if ($singleSend) {
                $singleSend->delete();
                $message = 'We have finished processing your request to delete single send';
                $this->emailNotification($message, route('user.singleSendList'), 'View your single send', $user->id);

                $route = route('user.singleSendList');
                $this->userActivity(
                    "You have finished processing to delete <a href='$route' class='text-primary'>$singleSend->single_send_name</a> single send!",
                    $user->id, $request->ip()
                );

                return response()->json($this->withSuccess('Deleted Successfully'));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function singleSendBulkDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($request->strIds == null) {
            return response()->json($this->withErrors('You do not send id'));
        } else {
            SingleSend::where('user_id', $user->id)->whereIn('id', $request->strIds)->where('status', '!=', 2)->get()->map(function ($query) {
                $query->delete();
            });
            $message = 'We have finished processing your request to delete multiple single send';
            $this->emailNotification($message, route('user.singleSendList'), 'View your single send', $user->id);

            $this->userActivity(
                "You have finished processing to multiple delete single send!", $user->id, $request->ip()
            );

            return response()->json($this->withSuccess('Deleted Successfully'));
        }
    }

    public function getTemplate(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['templates'] = EmailTemplate::where('user_id', null)->orWhere('user_id', $user->id)->get();
            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function getDynamicContent(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['contents'] = collect(CustomField::where('user_id', $user->id)->orWhere('user_id', null)->pluck('field_name'));
            $data['contents']->push('sender_name', 'sender_address', 'unsubscribe_preferences');
            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record not found'));
    }
}
