<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\SenderSaveRequest;
use App\Mail\SenderVerifyMail;
use App\Models\City;
use App\Models\Country;
use App\Models\Sender;
use App\Models\State;
use App\Traits\Activity;
use App\Traits\ApiValidation;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;

class SenderController extends Controller
{
    use ApiValidation, Activity;

    public function index(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['senders'] = Sender::where('user_id', $user->id)->latest()->get();
            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function senderSave(SenderSaveRequest $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $sender = new Sender();
            $fillData = $request->only($sender->getFillable());
            $fillData['user_id'] = $user->id;
            $fillData['utr'] = uniqid();
            $fillData['send_at'] = Carbon::now();
            $sender->fill($fillData)->save();
            Mail::to($sender->from_email)->send(new SenderVerifyMail($sender));

            $message = 'We have finished processing your request to add single sender';
            $this->emailNotification($message, route('user.senderList'), 'View your senders', $user->id);

            $route = route('user.senderList');
            $this->userActivity(
                "You have finished processing to create <a href='$route' class='text-primary'>$sender->from_name</a> single sender!",
                $user->id, $request->ip()
            );

            return response()->json($this->withSuccess('Added Successfully'));
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function senderDetails(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['sender'] = Sender::where('user_id', $user->id)->find($request->sender_id);
            if ($data['sender']) {
                return response()->json($this->withSuccess($data));
            }
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function senderUpdate(SenderSaveRequest $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $sender = Sender::where('user_id', $user->id)->find($request->sender_id);
            if ($sender) {
                $oldEmail = $sender->from_email;
                $fillData = $request->only($sender->getFillable());
                $sender->fill($fillData)->save();

                if ($oldEmail != $fillData['from_email']) {
                    $sender->is_verified = 0;
                    $sender->send_at = Carbon::now();
                    $sender->save();
                    Mail::to($fillData['from_email'])->send(new SenderVerifyMail($sender));
                }

                $route = route('user.senderList');
                $this->userActivity(
                    "You have finished processing to update <a href='$route' class='text-primary'>$sender->from_name</a> single sender!",
                    $user->id, $request->ip()
                );
                return response()->json($this->withSuccess('Updated Successfully'));
            }
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function senderSendMail(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $sender = Sender::where('user_id', $user->id)->find($request->sender_id);
            if ($sender) {
                try {
                    $sender->send_at = Carbon::now();
                    $sender->save();
                    Mail::to($sender->from_email)->send(new SenderVerifyMail($sender));

                    $route = route('user.senderList');
                    $this->userActivity(
                        "You have finished processing to send <a href='$route' class='text-primary'>$sender->from_name</a> Verification Mail!",
                        $user->id, $request->ip()
                    );

                    return response()->json($this->withSuccess('Verification Mail Send'));
                } catch (\Exception $e) {
                    return response()->json($this->withErrors($e->getMessage()));
                }
            }
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function senderDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $sender = Sender::where('user_id', $user->id)->find($request->sender_id);
            if ($sender) {
                $sender->delete();
                $message = 'We have finished processing your request to delete single sender';
                $this->emailNotification($message, route('user.senderList'), 'View your senders', $user->id);

                $route = route('user.senderList');
                $this->userActivity(
                    "You have finished processing to delete <a href='$route' class='text-primary'>$sender->from_name</a> single sender!",
                    $user->id, $request->ip()
                );

                return response()->json($this->withSuccess('Deleted Successfully'));
            }
        }
        return response()->json($this->withErrors('Record Not Found'));
    }

    public function getCountryStateCity(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['countries'] = Country::get();
            $data['states'] = State::get();
            $data['cities'] = City::get();

            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record Not Found'));
    }
}
