<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\SegmentEmailStoreRequest;
use App\Models\Segment;
use App\Models\SegmentEmail;
use App\Traits\Activity;
use App\Traits\ApiValidation;
use Illuminate\Http\Request;

class SegmentController extends Controller
{
    use ApiValidation, Activity;

    public function index(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['segments'] = Segment::where('user_id', $user->id)->withCount('segmentEmails')
                ->orderBy('id', 'desc')
                ->get();
            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function segmentEmailList(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $segment = Segment::where('user_id', $user->id)->find($request->segment_id);
            if ($segment) {
                $data['contactEmails'] = SegmentEmail::where('user_id', $user->id)->where('segment_id', $segment->id)
                    ->orderBy('id', 'desc')->get();
                return response()->json($this->withSuccess($data));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function segmentEmailStore(SegmentEmailStoreRequest $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        $segment = Segment::where('user_id', $user->id)->find($request->segment_id);
        if ($user && $segment) {
            $segmentEmail = new SegmentEmail();
            $fillData = $request->only($segmentEmail->getFillable());
            $fillData['user_id'] = $user->id;
            $fillData['segment_id'] = $request->segment_id;
            $segmentEmail->fill($fillData)->save();

            $this->automationTriggered($segmentEmail ?? [], 'segments_', $segmentEmail->segment_id, $user->id);

            $this->userActivity("A new segment email was created!", $user->id, $request->ip());

            return response()->json($this->withSuccess('Added Successfully'));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function segmentEmailDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $segment = SegmentEmail::where('user_id', $user->id)->find($request->segment_recipient_id);
            if ($segment) {
                $segment->delete();
                $message = 'We have finished processing your request to delete segment emails';
                $this->emailNotification($message, null, null, $user->id);
                $this->userActivity("You have finished processing to delete segment emails!", $user->id, $request->ip());
                return response()->json($this->withSuccess('Deleted Successfully'));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function segmentDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $segment = Segment::where('user_id', $user->id)->find($request->segment_id);
            if ($segment) {
                $segment->delete();
                $message = 'We have finished processing your request to delete single segment';
                $this->emailNotification($message, route('user.segmentList'), 'View your segments', $user->id);

                $route = route('user.segmentList');
                $this->userActivity(
                    "You have finished processing to delete <a href='$route' class='text-primary'>$segment->segment_name</a> segment!",
                    $user->id, $request->ip()
                );

                return response()->json($this->withSuccess('Deleted Successfully'));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }
}
