<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\ContactTypeRequest;
use App\Models\Contact;
use App\Models\ContactType;
use App\Models\Country;
use App\Models\CustomField;
use App\Traits\Activity;
use App\Traits\ApiValidation;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class ContactController extends Controller
{
    use ApiValidation, Activity;

    public function contactTypeList(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['contactTypes'] = ContactType::where('user_id', $user->id)->withCount('contacts')
                ->get();
            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function contactTypeSave(ContactTypeRequest $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $contactType = new ContactType();
            $fillData = $request->only($contactType->getFillable());
            $fillData['user_id'] = $user->id;
            $contactType->fill($fillData)->save();
            $this->userActivity("You have finished processing to create contact type!", $user->id, $request->ip());
            return response()->json($this->withSuccess('Added Successfully'));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function contactTypeView(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['contactType'] = ContactType::where('user_id', $user->id)
                ->find($request->contact_type_id);
            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function contactTypeUpdate(ContactTypeRequest $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $contactType = ContactType::where('user_id', $user->id)->find($request->id);
            if ($contactType) {
                $fillData = $request->only($contactType->getFillable());
                $contactType->fill($fillData)->save();
                $this->userActivity("You have finished processing to update contact type!", $user->id, $request->ip());
                return response()->json($this->withSuccess('Updated Successfully'));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function contactTypeDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $contactType = ContactType::where('user_id', $user->id)->find($request->contact_type_id);
            if ($contactType) {
                $contactType->delete();
                $message = 'We have finished processing your request to delete contact type';
                $this->emailNotification($message, route('user.contactTypeList'), 'View your contact type', $user->id);
                $this->userActivity(
                    "You have finished processing to delete contact type!", $user->id, $request->ip()
                );
                return response()->json($this->withSuccess('Deleted Successfully'));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function contactList(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['contacts'] = Contact::where('user_id', $user->id)->with('contactType')
                ->orderBy('id', 'desc')
                ->get();
            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function contactAdd(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['customFields'] = CustomField::where('user_id', $user->id)->get();
            $validator = Validator::make($request->all(), [
                'contact_type_id' => 'required|numeric',
                'country' => 'required',
                'email' => 'required|email'
            ], [
                'contact_type_id' => 'Contact Type field is required',
            ]);
            if ($validator->fails()) {
                return response()->json($this->withErrors($validator->errors()));
            }

            $contact = new Contact();
            $fillData = $request->only($contact->getFillable());
            $fillData['user_id'] = $user->id;
            $custom_field = [];
            if ($data['customFields']) {
                foreach ($data['customFields'] as $item) {
                    $custom_field[$item->field_name] = [
                        'field_name' => removeHyphenInString(ucfirst($item->field_name)),
                        'field_value' => $request->{$item->field_name},
                    ];
                }
            }
            $fillData['custom_field'] = $custom_field;
            $contact->fill($fillData)->save();
            updateWallet($user->id, 1, 'use_contacts', 1);

            $message = 'We have finished processing your request to create contact';
            $this->emailNotification($message, route('user.contactList', $request->contact_type_id), 'View your contacts', $user->id);

            $recipients = [$contact];
            $this->automationTriggered($recipients, 'contacts_', $request->contact_type_id, $user->id);

            $route = route('user.contactList', $request->contact_type_id);
            $this->userActivity(
                "You have finished processing to create contact <a href='$route' class='text-primary'>$contact->email</a>!",
                $user->id, $request->ip()
            );

            return response()->json($this->withSuccess('Contact Added Successfully'));
        }

        return response()->json($this->withErrors('Record not found'));
    }

    public function contactCsvSample()
    {
        $file = 'contacts-sample.csv';
        $full_path = 'assets/' . $file;
        $title = 'contacts-upload-sample';
        $ext = pathinfo($file, PATHINFO_EXTENSION);
        $mimetype = mime_content_type($full_path);
        header('Content-Disposition: attachment; filename="' . $title . '.' . $ext . '";');
        header("Content-Type: " . $mimetype);
        return readfile($full_path);
    }

    public function contactUploadCsv(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $validator = Validator::make($request->all(), [
                'contact_type_id' => 'required',
                'file' => 'required'
            ]);
            if ($validator->fails()) {
                return response()->json($this->withErrors($validator->errors()));
            }

            $contactType = ContactType::where('user_id', $user->id)->orWhere('user_id', null)
                ->find($request->contact_type_id);

            if ($contactType) {
                if ($request->file->getClientOriginalExtension() != 'csv') {
                    return response()->json($this->withErrors('Only accepted .csv files'));
                }
                $file = fopen($request->file->getRealPath(), 'r');

                $firstIteration = true;
                $count = 0;
                while ($csvLine = fgetcsv($file)) {
                    if ($firstIteration) {
                        $firstIteration = false;
                        continue; // Skip the first iteration
                    }
                    $contacts[] = Contact::firstOrCreate(
                        [
                            'email' => $csvLine[2],
                            'contact_type_id' => $contactType->id,
                        ],
                        [
                            'user_id' => $user->id,
                            'first_name' => $csvLine[0],
                            'last_name' => $csvLine[1],
                            'alt_email' => $csvLine[3],
                            'country' => $csvLine[4],
                            'state' => $csvLine[5],
                            'city' => $csvLine[6],
                            'address_line1' => $csvLine[7],
                            'address_line2' => $csvLine[8],
                            'state_province_region' => $csvLine[9],
                            'postal_code' => $csvLine[10],
                        ]
                    );
                    $count++;
                }
                updateWallet($user->id, $count, 'use_contacts', 1);

                $message = 'We have finished processing your request to upload contacts CSV';
                $this->emailNotification($message, route('user.contactList', $contactType->id), 'View your contacts', $user->id);

                $this->automationTriggered($contacts ?? [], 'contacts_', $contactType->id, $user->id);

                $route = route('user.contactList', $contactType->id);
                $this->userActivity(
                    "You have finished processing to upload <a href='$route' class='text-primary'>contacts CSV</a>!",
                    $user->id, $request->ip()
                );

                return response()->json($this->withSuccess('Imported Successfully'));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function contactDetail(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['contact'] = Contact::where('user_id', $user->id)->find($request->contact_id);
            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function contactUpdate(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $contact = Contact::where('user_id', $user->id)->find($request->contact_id);
            if ($contact) {
                $data['customFields'] = CustomField::where('user_id', $user->id)->get();
                $validator = Validator::make($request->all(), [
                    'contact_type_id' => 'required|numeric',
                    'country' => 'required',
                    'email' => 'required|email'
                ], [
                    'contact_type_id' => 'Contact Type field is required',
                ]);
                if ($validator->fails()) {
                    return response()->json($this->withErrors($validator->errors()));
                }

                $custom_field = [];
                $fillData = $request->only($contact->getFillable());
                if ($data['customFields']) {
                    foreach ($data['customFields'] as $item) {
                        $custom_field[$item->field_name] = [
                            'field_name' => removeHyphenInString(ucfirst($item->field_name)),
                            'field_value' => $request->{$item->field_name},
                        ];
                    }
                }
                $fillData['custom_field'] = $custom_field;
                $contact->fill($fillData)->save();

                $this->userActivity(
                    "You have finished processing to update contact!", $user->id, $request->ip()
                );

                return response()->json($this->withSuccess('Contact Updated Successfully'));
            }
        }

        return response()->json($this->withErrors('Record not found'));
    }

    public function contactDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $contact = Contact::where('user_id', $user->id)->find($request->contact_id);
            if ($contact) {
                $contact->delete();
                $message = 'We have finished processing your request to delete single contact';
                $this->emailNotification($message, route('user.contactList', $contact->contact_type_id), 'View your contacts', $user->id);

                $route = route('user.contactList', $contact->contact_type_id);
                $this->userActivity(
                    "You have finished processing to delete <a href='$route' class='text-primary'>single contact</a>!",
                    $user->id, $request->ip()
                );

                return response()->json($this->withSuccess('Deleted Successfully'));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function contactBulkDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($request->strIds == null) {
            return response()->json($this->withErrors('You do not select ID.'));
        } else {
            Contact::where('user_id', $user->id)->whereIn('id', $request->strIds)->get()->map(function ($query) {
                $query->delete();
            });
            $message = 'We have finished processing your request to delete multiple contact';
            $this->emailNotification($message, null, null, $user->id);

            $this->userActivity(
                "You have finished processing to delete multiple contact!", $user->id, $request->ip()
            );

            return response()->json($this->withSuccess('Deleted Successfully'));
        }
    }
}
