<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\AutomationStoreRequest;
use App\Models\Automation;
use App\Models\AutomationSchedule;
use App\Models\EmailTemplate;
use App\Models\Sender;
use App\Traits\Activity;
use App\Traits\ApiValidation;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Ramsey\Uuid\Uuid;

class AutomationController extends Controller
{
    use ApiValidation, Activity;

    public function index(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['automations'] = Automation::where('user_id', $user->id)->withCount(['automationSchedule'])->orderBy('id', 'desc')
                ->get();
            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function automationDuplicate(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $automation = Automation::where('user_id', $user->id)->with(['automationSchedule'])
                ->where('utr', $request->utr)->first();
            if ($automation) {
                try {
                    $fillData = new Automation($automation->attributesToArray());
                    $fillData['utr'] = (Uuid::uuid1())->toString();
                    $fillData['status'] = 0;
                    $fillData['live_at'] = null;
                    $fillData['total_triggered'] = 0;
                    $fillData['total_delivered'] = 0;
                    $fillData['total_unique_opens'] = 0;
                    $fillData['total_opens'] = 0;
                    $fillData['total_bounces'] = 0;
                    $fillData['total_unsubscribe'] = 0;
                    $fillData['created_at'] = Carbon::now();
                    $fillData['updated_at'] = Carbon::now();
                    $fillData->save();

                    if (count($automation->automationSchedule) > 0) {
                        foreach ($automation->automationSchedule as $schedule) {
                            $scheduleData = new AutomationSchedule($schedule->attributesToArray());
                            $scheduleData['automation_id'] = $fillData->id;
                            $scheduleData->save();
                        }
                    }

                    $message = 'We have finished processing your request to create duplicate automation';
                    $this->emailNotification($message, route('user.automation'), 'View your automation', $user->id);

                    $route = route('user.automationShow', $fillData->utr);
                    $this->userActivity(
                        "You have finished processing to create duplicate <a href='$route' class='text-primary'>$automation->automation_name</a> automation!",
                        $user->id, $request->ip()
                    );

                    $data['utr'] = $fillData->utr;
                    return response()->json($this->withSuccess($data));
                } catch (\Exception $e) {
                    return response()->json($this->withErrors($e->getMessage()));
                }
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function automationDetails(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $data['automation'] = Automation::where('user_id', $user->id)
                ->where('utr', $request->utr)->first();
            return response()->json($this->withSuccess($data));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function automationSchedule(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $automation = Automation::where('user_id', $user->id)->where('utr', $request->utr)->first();
            if ($automation) {
                $data['automation_schedules'] = AutomationSchedule::where('user_id', $user->id)
                    ->where('automation_id', $automation->id)->get();
                return response()->json($this->withSuccess($data));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function automationPause(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $automation = Automation::where('user_id', $user->id)->where('utr', $request->utr)->where('status', 1)->first();
            if ($automation) {
                try {
                    $automation->status = 0;
                    $automation->save();
                    $message = 'We have finished processing your request to paused automation';
                    $this->emailNotification($message, route('user.automation'), 'View your automation', $user->id);

                    $route = route('user.automationShow', $automation->utr);
                    $this->userActivity(
                        "You have finished processing to paused <a href='$route' class='text-primary'>$automation->automation_name</a> automation!",
                        $user->id, $request->ip()
                    );

                    return response()->json($this->withSuccess('Automation Paused Successfully'));
                } catch (\Exception $e) {
                    return response()->json($this->withErrors('something went wrong.Please try again'));
                }
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function automationSetLive(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $automation = Automation::where('user_id', $user->id)->with(['automationSchedule'])
                ->where('utr', $request->utr)->first();

            if (!$automation->automation_name) {
                return response()->json($this->withErrors('Automation name require to live automation'));
            } elseif (!$automation->automation_for) {
                return response()->json($this->withErrors('Automation targeting recipient require to live automation'));
            } elseif (!$automation->unsubscribe_group_id) {
                return response()->json($this->withErrors('Unsubscribe group require to live automation'));
            }

            if ($automation) {
                if (count($automation->automationSchedule) > 0) {
                    foreach ($automation->automationSchedule as $schedule) {
                        if (!$schedule->template_html) {
                            return response()->json($this->withErrors('Please select email format for all emails'));
                        }
                    }
                } else {
                    return response()->json($this->withErrors('Please set email first for automation live'));
                }

                $automation->status = 1;
                $automation->live_at = Carbon::now();
                $automation->save();

                $message = 'We have finished processing your request to create automation and set live';
                $this->emailNotification($message, route('user.automation'), 'View your automation', $user->id);

                $route = route('user.automationShow', $automation->utr);
                $this->userActivity(
                    "You have finished processing to create <a href='$route' class='text-primary'>$automation->automation_name</a> automation and set live!",
                    $user->id, $request->ip()
                );

                return response()->json($this->withSuccess('Automation has been set live'));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function automationDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        $automation = Automation::where('user_id', $user->id)->where('utr', $request->utr)->first();
        if ($automation) {
            $automation->delete();
            $message = 'We have finished processing your request to delete automation';
            $this->emailNotification($message, route('user.automation'), 'View your automation', $user->id);

            $this->userActivity("You have finished processing to delete automation!", $user->id, $request->ip());

            return response()->json($this->withSuccess('Deleted Successfully'));
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function automationBulkDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            if ($request->strIds == null) {
                return response()->json($this->withErrors('ID not found'));
            } else {
                Automation::where('user_id', $user->id)->whereIn('id', $request->strIds)->get()->map(function ($query) {
                    $query->delete();
                });
                $message = 'We have finished processing your request to delete multiple automation';
                $this->emailNotification($message, route('user.automation'), 'View your automation', $user->id);

                $this->userActivity("You have finished processing to delete multiple automation!", $user->id, $request->ip());

                return response()->json($this->withSuccess('Deleted Successfully'));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function automationCreate(AutomationStoreRequest $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            if ($request->type == 'custom') {
                $automation = new Automation();
                $automation->user_id = $user->id;
                $getData = getIdAndTable($request->automation_for);
                $automation->automation_for = $getData['id'];
                $automation->automation_for_table = $getData['table'];
                $automation->automation_name = $request->automation_name;
                $automation->unsubscribe_group_id = $request->unsubscribe_group_id;
                $automation->automation_type = 'custom';
                $automation->utr = (Uuid::uuid1())->toString();
                $automation->status = 0;
                $automation->save();

                $data['utr'] = $automation->utr;

                $route = route('user.automationShow', $automation->utr);
                $this->userActivity(
                    "A draft automation <a href='$route' class='text-primary'>$automation->automation_name</a> was created!",
                    $user->id, $request->ip()
                );

                return response()->json($this->withSuccess($data));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function automationScheduleCreate(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'utr' => 'required',
            'sender_id' => 'required|integer',
            'subject' => 'required',
            'waiting_period' => 'required',
            'waiting_type' => 'required|in:instantly,days,hours',
            'template_id' => 'required',
            'preview_img' => 'required',
            'template_json' => 'required_if:template_html,null|required_if:template_id,1',
            'template_html' => 'required_if:template_json,null|required_if:template_id,1',
        ]);

        if ($validator->fails()) {
            return response()->json($this->withErrors($validator->errors()));
        }

        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $automation = Automation::where('user_id', $user->id)->where('utr', $request->utr)->first();
            $sender = Sender::where('user_id', $user->id)->where('is_verified', 1)->find($request->sender_id);
            if (!$sender) {
                return response()->json($this->withErrors('Sender not found'));
            }
            $template = EmailTemplate::find($request->template_id);
            if ($automation) {
                AutomationSchedule::create([
                    'user_id' => $user->id,
                    'automation_id' => $automation->id,
                    'sender_id' => $sender->id ?? null,
                    'subject' => $request->subject ?? 'Welcome',
                    'waiting_period' => $request->waiting_period,
                    'waiting_type' => $request->waiting_type,
                    'template_id' => $request->template_id,
                    'preview_img' => $request->preview_img,
                    'template_json' => $request->template_html ? '{}' : $request->template_json ?? $template->json_code,
                    'template_html' => $request->template_html ?? $template->html_code,
                    'utr' => (Uuid::uuid1())->toString(),
                ]);

                $this->userActivity("A automation schedule was added!", $user->id, $request->ip());
                return response()->json($this->withSuccess('Schedule Added'));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }

    public function automationScheduleDelete(Request $request)
    {
        $user = $this->user($request->header('publicKey'), $request->header('SecretKey'), $request->ip());
        if ($user) {
            $automationSchedule = AutomationSchedule::where('user_id', $user->id)->where('utr', $request->utr)->first();
            if ($automationSchedule) {
                $automationSchedule->delete();

                $this->userActivity("A new automation schedule was deleted!", $user->id, $request->ip());

                return response()->json($this->withSuccess('Deleted Successfully'));
            }
        }
        return response()->json($this->withErrors('Record not found'));
    }
}
