<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\PrepaidCreditStoreRequest;
use App\Models\PrepaidCredit;
use Google\Service\AndroidPublisher\PrepaidPlan;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class PrepaidCreditEmailChecker extends Controller
{
    public function index()
    {
        $data['planRecord'] = collect(PrepaidCredit::selectRaw('COUNT(id) AS totalPlan')
            ->selectRaw('COUNT(CASE WHEN status = 1 THEN id END) AS activePlan')
            ->selectRaw('(COUNT(CASE WHEN status = 1 THEN id END) / COUNT(id)) * 100 AS activePlanPercentage')
            ->selectRaw('COUNT(CASE WHEN status = 0 THEN id END) AS inActivePlan')
            ->selectRaw('(COUNT(CASE WHEN status = 0 THEN id END) / COUNT(id)) * 100 AS inActivePlanPercentage')
            ->selectRaw('COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) AS todayPlan')
            ->selectRaw('(COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) / COUNT(id)) * 100 AS todayPlanPercentage')
            ->selectRaw('COUNT(CASE WHEN MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE()) THEN id END) AS thisMonthPlan')
            ->selectRaw('(COUNT(CASE WHEN MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE()) THEN id END) / COUNT(id)) * 100 AS thisMonthPlanPercentage')
            ->get()
            ->toArray())->collapse();
        return view('admin.prepaid_credits.index', $data);
    }

    public function prepaidCreditsSearch(Request $request)
    {
        $prepaidCredits = PrepaidCredit::
        when(!empty($request->search['value']), function ($query) use ($request) {
            $query->where(function ($subquery) use ($request) {
                $subquery->where('credits', 'LIKE', '%' . $request->search['value'] . '%')
                    ->orWhere('price', 'LIKE', '%' . $request->search['value'] . '%')
                    ->orWhere('per_credit_price', 'LIKE', '%' . $request->search['value'] . '%');
            });
        })->orderBy('id', 'DESC');


        return DataTables::of($prepaidCredits)
            ->addColumn('no', function () {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('credits', function ($item) {
                return $item->credits . ' Credits';
            })
            ->addColumn('price', function ($item) {
                return basicControl()->currency_symbol . getAmount($item->price, 2);
            })
            ->addColumn('price_per_credit', function ($item) {
                return basicControl()->currency_symbol . getAmount($item->per_credit_price, 5);
            })
            ->addColumn('status', function ($item) {
                if ($item->status == 1) {
                    return '<span class="badge bg-soft-success text-success">
                    <span class="legend-indicator bg-success"></span>' . trans('Active') . '
                  </span>';

                } else {
                    return '<span class="badge bg-soft-danger text-danger">
                    <span class="legend-indicator bg-danger"></span>' . trans('In Active') . '
                  </span>';
                }
            })
            ->addColumn('created_at', function ($item) {
                return dateTime($item->created_at, basicControl()->date_time_format);

            })
            ->addColumn('action', function ($item) {
                $update = route('admin.prepaidCreditUpdate', $item->id);
                $delete = route('admin.prepaidCreditDelete', $item->id);
                return '<div class="btn-group" role="group">
                      <a href="' . $update . '" class="btn btn-white btn-sm">
                        <i class="fal fa-edit me-1"></i> ' . trans("Update") . '
                      </a>
                    <div class="btn-group">
                      <button type="button" class="btn btn-white btn-icon btn-sm dropdown-toggle dropdown-toggle-empty" id="userEditDropdown" data-bs-toggle="dropdown" aria-expanded="false"></button>
                      <div class="dropdown-menu dropdown-menu-end mt-1" aria-labelledby="userEditDropdown">
                        <a class="dropdown-item delete_btn" href="javascript:void(0)" data-bs-target="#delete"
                           data-bs-toggle="modal" data-route="' . $delete . '">
                          <i class="fal fa-trash dropdown-item-icon"></i> ' . trans("Delete") . '
                       </a>
                      </div>
                    </div>
                  </div>';
            })
            ->rawColumns(['credits', 'price', 'price_per_credit', 'status', 'created_at', 'action'])
            ->make(true);
    }


    public function prepaidCreditCreate(PrepaidCreditStoreRequest $request)
    {
        if ($request->method() == 'GET') {
            return view('admin.prepaid_credits.create');
        } elseif ($request->method() == 'POST') {
            $prepaidCredit = new PrepaidCredit();
            $fillData = $request->only($prepaidCredit->getFillable());
            $fillData['per_credit_price'] = $fillData['price'] / $fillData['credits'];
            $prepaidCredit->fill($fillData)->save();
            return back()->with('success', 'Created Successfully');
        }
    }

    public function prepaidCreditUpdate(PrepaidCreditStoreRequest $request, $id)
    {
        $prepaidCredit = PrepaidCredit::findOrFail($id);
        if ($request->method() == 'GET') {
            return view('admin.prepaid_credits.edit', compact('prepaidCredit'));
        } elseif ($request->method() == 'POST') {
            $fillData = $request->only($prepaidCredit->getFillable());
            $fillData['per_credit_price'] = $fillData['price'] / $fillData['credits'];
            $prepaidCredit->fill($fillData)->save();
            return back()->with('success', 'Updated Successfully');
        }
    }

    public function prepaidCreditDelete($id)
    {
        $prepaidCredit = PrepaidCredit::findOrFail($id);
        $prepaidCredit->delete();
        return back()->with('success', 'Deleted Successfully');
    }
}
