<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\PlanStoreRequest;
use App\Jobs\gateway\paddle;
use App\Models\Deposit;
use App\Models\Gateway;
use App\Models\Plan;
use App\Models\SubscriptionPurchase;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Queue\Jobs\Job;
use Yajra\DataTables\Facades\DataTables;
use Facades\App\Services\Subscription\square\Payment;

class PlanController extends Controller
{
    public function plan()
    {
        $data['planRecord'] = Plan::selectRaw('COUNT(id) AS totalPlan')
            ->selectRaw('COUNT(CASE WHEN status = 1 THEN id END) AS activePlan')
            ->selectRaw('(COUNT(CASE WHEN status = 1 THEN id END) / COUNT(id)) * 100 AS activePlanPercentage')
            ->selectRaw('COUNT(CASE WHEN status = 0 THEN id END) AS inActivePlan')
            ->selectRaw('(COUNT(CASE WHEN status = 0 THEN id END) / COUNT(id)) * 100 AS inActivePlanPercentage')
            ->selectRaw('COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) AS todayPlan')
            ->selectRaw('(COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) / COUNT(id)) * 100 AS todayPlanPercentage')
            ->selectRaw('COUNT(CASE WHEN MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE()) THEN id END) AS thisMonthPlan')
            ->selectRaw('(COUNT(CASE WHEN MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE()) THEN id END) / COUNT(id)) * 100 AS thisMonthPlanPercentage')
            ->get()
            ->toArray();

        return view('admin.plan.list', $data);
    }

    public function planSearch(Request $request)
    {
        $contacts = Plan::
        when(!empty($request->search['value']), function ($query) use ($request) {
            $query->where(function ($subquery) use ($request) {
                $subquery->where('plan_name', 'LIKE', '%' . $request->search['value'] . '%')
                    ->orWhere('payment_frequency', 'LIKE', '%' . $request->search['value'] . '%')
                    ->orWhere('price', 'LIKE', '%' . $request->search['value'] . '%')
                    ->orWhere('number_of_contacts', 'LIKE', '%' . $request->search['value'] . '%')
                    ->orWhere('number_of_emails', 'LIKE', '%' . $request->search['value'] . '%');
            });
        })->orderBy('id', 'DESC');


        return DataTables::of($contacts)
            ->addColumn('no', function () {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('plan_name', function ($item) {
                return $item->plan_name;
            })
            ->addColumn('frequency', function ($item) {
                if (!$item->is_free) {
                    return ucfirst($item->payment_frequency);
                } else {
                    return '-';
                }
            })
            ->addColumn('price', function ($item) {
                if (!$item->is_free) {
                    return basicControl()->currency_symbol . $item->price;
                } else {
                    return '-';
                }
            })
            ->addColumn('contacts', function ($item) {
                return 'Upto ' . $item->number_of_contacts;
            })
            ->addColumn('emails', function ($item) {
                return 'Upto ' . $item->number_of_emails;
            })
            ->addColumn('status', function ($item) {
                if ($item->status == 1) {
                    return '<span class="badge bg-soft-success text-success">
                    <span class="legend-indicator bg-success"></span>' . trans('Active') . '
                  </span>';

                } else {
                    return '<span class="badge bg-soft-danger text-danger">
                    <span class="legend-indicator bg-danger"></span>' . trans('In Active') . '
                  </span>';
                }
            })
            ->addColumn('created_at', function ($item) {
                if (!$item->is_free) {
                    return dateTime($item->created_at, basicControl()->date_time_format);
                } else {
                    return '-';
                }

            })
            ->addColumn('action', function ($item) {
                $update = route('admin.planEdit', $item->id);
                return '<div class="btn-group" role="group">
                      <a href="' . $update . '"
                       class="btn btn-white btn-sm delete_btn">
                        <i class="fal fa-edit me-1"></i> ' . trans("Update") . '
                      </a>
                  </div>';
            })
            ->rawColumns(['plan_name', 'frequency', 'price', 'contacts', 'emails', 'status', 'created_at', 'action'])
            ->make(true);
    }

    public function planCreate()
    {
        return view('admin.plan.create');
    }

    public function planStore(PlanStoreRequest $request)
    {
        $plan = new Plan();
        $fillData = $request->only($plan->getFillable());
        $plan->fill($fillData)->save();

        $gateways = Gateway::where('subscription_on', 1)->get();
        foreach ($gateways as $gateway) {
            $jobs = 'App\\Jobs\\gateway\\' . $gateway->code;
            $jobs::dispatch($plan, $gateway, 'create');
        }

        return back()->with('success', 'Plan Created Successfully');
    }

    public function planEdit($id)
    {
        $data['plan'] = Plan::findOrFail($id);
        $data['gateways'] = Gateway::select(['id', 'name', 'code', 'subscription_on'])
            ->where('subscription_on', 1)->get();
        return view('admin.plan.edit', $data);
    }

    public function planUpdate(PlanStoreRequest $request, $id)
    {
        $subscriptionPlan = Plan::findOrFail($id);

        $arr = [];
        if ($request->gateway_plan_id && count($request->gateway_plan_id) > 0) {
            foreach ($request->gateway_plan_id as $key => $planId) {
                $arr[$key] = $planId[0];
            }
        }

        $oldSubscription = $subscriptionPlan;
        $fillData = $request->only($subscriptionPlan->getFillable());
        $fillData['payment_frequency'] = $subscriptionPlan->payment_frequency;
        $fillData['gateway_plan_id'] = $arr;
        $subscriptionPlan->fill($fillData)->save();

        if ($request->price != $oldSubscription->price && !$subscriptionPlan->is_free) {
            $gateways = Gateway::where('subscription_on', 1)->get();
            foreach ($gateways as $gateway) {
                $jobs = 'App\\Jobs\\gateway\\' . $gateway->code;
                $jobs::dispatch($subscriptionPlan, $gateway, 'update');
            }
        }

        if ($request->status != $oldSubscription->status && !$subscriptionPlan->is_free) {
            if ($request->status == 1) {
                $gateways = Gateway::where('subscription_on', 1)->get();
                foreach ($gateways as $gateway) {
                    $jobs = 'App\\Jobs\\gateway\\' . $gateway->code;
                    $jobs::dispatch($subscriptionPlan, $gateway, 'active');
                }
            } elseif ($request->status == 0 && !$subscriptionPlan->is_free) {
                $gateways = Gateway::where('subscription_on', 1)->get();
                foreach ($gateways as $gateway) {
                    $jobs = 'App\\Jobs\\gateway\\' . $gateway->code;
                    $jobs::dispatch($subscriptionPlan, $gateway, 'deactive');
                }
            }
        }

        return back()->with('success', 'Plan has been updated');
    }

    public function subscriptionList()
    {
        $data['subscriptionRecord'] = collect(SubscriptionPurchase::where('payment_status', 1)->selectRaw('COUNT(id) AS totalSubscription')
            ->selectRaw('COUNT(CASE WHEN status = 1 THEN id END) AS activeSubscription')
            ->selectRaw('(COUNT(CASE WHEN status = 1 THEN id END) / COUNT(id)) * 100 AS activeSubscriptionPercentage')
            ->selectRaw('COUNT(CASE WHEN status = 0 THEN id END) AS inActiveSubscription')
            ->selectRaw('(COUNT(CASE WHEN status = 0 THEN id END) / COUNT(id)) * 100 AS inActiveSubscriptionPercentage')
            ->selectRaw('COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) AS todaySubscription')
            ->selectRaw('(COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) / COUNT(id)) * 100 AS todaySubscriptionPercentage')
            ->selectRaw('COUNT(CASE WHEN MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE()) THEN id END) AS thisMonthSubscription')
            ->selectRaw('(COUNT(CASE WHEN MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE()) THEN id END) / COUNT(id)) * 100 AS thisMonthSubscriptionPercentage')
            ->get()
            ->toArray())->collapse();

        return view('admin.plan.subscription-list', $data);
    }

    public function subscriptionListSearch(Request $request)
    {
        $subscriptions = SubscriptionPurchase::
        when(!empty($request->search['value']), function ($query) use ($request) {
            $query->where(function ($subquery) use ($request) {
                $subquery->where('plan_name', 'LIKE', '%' . $request->search['value'] . '%')
                    ->orWhere('frequency', 'LIKE', '%' . $request->search['value'] . '%')
                    ->orWhere('price', 'LIKE', '%' . $request->search['value'] . '%')
                    ->orWhereHas('gateway', function ($userQuery) use ($request) {
                        $userQuery->where('name', 'LIKE', '%' . $request->search['value'] . '%');
                    })
                    ->orWhereHas('user', function ($userQuery) use ($request) {
                        $userQuery->where('firstname', 'LIKE', '%' . $request->search['value'] . '%')
                            ->orWhere('lastname', 'LIKE', '%' . $request->search['value'] . '%')
                            ->orWhere('username', 'LIKE', '%' . $request->search['value'] . '%');
                    });
            });
        })->orderBy('id', 'DESC');

        return DataTables::of($subscriptions)
            ->addColumn('no', function () {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('plan_name', function ($item) {
                return $item->plan_name;
            })
            ->addColumn('frequency', function ($item) {
                return ucfirst($item->frequency);
            })
            ->addColumn('price', function ($item) {
                if (optional($item->subscriptionPlan)->is_free) {
                    return '-';
                } else {
                    return basicControl()->currency_symbol . $item->price;
                }
            })
            ->addColumn('paid_by', function ($item) {
                return optional($item->gateway)->name ?? '-';
            })
            ->addColumn('subscriber', function ($item) {
                $userProfileRoute = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="' . $userProfileRoute . '">
                                <div class="flex-shrink-0">
                                  ' . optional($item->user)->profilePicture() . '
                                </div>
                                <div class="flex-grow-1 ms-3">
                                  <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                                  <span class="fs-6 text-body">' . optional($item->user)->username . '</span>
                                </div>
                              </a>';
            })
            ->addColumn('subscription_type', function ($item) {
                if ($item->api_subscription_id) {
                    return '<span class="badge bg-soft-success text-success">
                    <span class="legend-indicator bg-success"></span>' . trans('Automatic') . '
                  </span>';

                } else {
                    return '<span class="badge bg-soft-warning text-warning">
                    <span class="legend-indicator bg-warning"></span>' . trans('Manual') . '
                  </span>';
                }
            })
            ->addColumn('status', function ($item) {
                if ($item->status == 1) {
                    return '<span class="badge bg-soft-success text-success">
                    <span class="legend-indicator bg-success"></span>' . trans('Running') . '
                  </span>';

                } else {
                    return '<span class="badge bg-soft-danger text-danger">
                    <span class="legend-indicator bg-danger"></span>' . trans('Cancel') . '
                  </span>';
                }
            })
            ->addColumn('subscribe_on', function ($item) {
                return dateTime($item->created_at, basicControl()->date_time_format);
            })
            ->addColumn('action', function ($item) {
                $cancel = route('admin.subscription.cancel', $item->id);
                if ($item->api_subscription_id && $item->status) {
                    return '<div class="btn-group" role="group">
                      <a href="javascript:void(0)"
                       class="btn btn-white btn-sm delete_btn" data-bs-target="#delete"
                           data-bs-toggle="modal" data-route="' . $cancel . '">
                        <i class="fal fa-times me-1"></i> ' . trans("Cancel") . '
                      </a>
                  </div>';
                } else {
                    return '-';
                }
            })
            ->rawColumns(['plan_name', 'frequency', 'price', 'paid_by', 'subscriber', 'subscription_type', 'status', 'subscribe_on', 'action'])
            ->make(true);
    }

    public function subsCancel($id)
    {
        $subscriptionPurchase = SubscriptionPurchase::findOrFail($id);

        try {
            $code = $subscriptionPurchase->gateway->code;
            $getwayObj = 'App\\Services\\Subscription\\' . $code . '\\Payment';
            $data = $getwayObj::cancelSubscription($subscriptionPurchase);
            if ($data['status'] == 'success') {
                $subscriptionPurchase->status = 0;
                $subscriptionPurchase->cancel_at = Carbon::now();
                $subscriptionPurchase->save();
                return back()->with('success', 'subscription has been canceled');
            } else {
                return back()->with('alert', 'You can not cancel subscription');
            }
        } catch (\Exception $e) {
            return back()->with('alert', 'Something went wrong. Please try again');
        }
    }
}
