<?php

namespace App\Http\Controllers\Admin\GeneratedHistory;

use App\Http\Controllers\Controller;
use App\Models\SingleSend;
use App\Models\SingleSendActivity;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;
use DateTime;

class SingleSendController extends Controller
{
    public function index()
    {
        $data['singleSendRecord'] = SingleSend::selectRaw('COUNT(id) AS totalSingleSend')
            ->selectRaw('COUNT(CASE WHEN status = 0 OR status = 1 THEN id END) AS initiateSingleSend')
            ->selectRaw('(COUNT(CASE WHEN status = 0 OR status = 1 THEN id END) / COUNT(id)) * 100 AS initiateSingleSendPercentage')
            ->selectRaw('COUNT(CASE WHEN status = 2 THEN id END) AS runningSingleSend')
            ->selectRaw('(COUNT(CASE WHEN status = 2 THEN id END) / COUNT(id)) * 100 AS runningSingleSendPercentage')
            ->selectRaw('COUNT(CASE WHEN status = 3 THEN id END) AS completeSingleSend')
            ->selectRaw('(COUNT(CASE WHEN status = 3 THEN id END) / COUNT(id)) * 100 AS completeSingleSendPercentage')
            ->selectRaw('COUNT(CASE WHEN schedule_type = 1 THEN id END) AS scheduleSingleSend')
            ->selectRaw('(COUNT(CASE WHEN schedule_type = 1 THEN id END) / COUNT(id)) * 100 AS scheduleSingleSendPercentage')
            ->get()
            ->toArray();

        return view('admin.single-send.list', $data);
    }

    public function singleSendSearch(Request $request)
    {
        $filterName = $request->name;
        $filterCreator = $request->creator;
        $filterStatus = $request->filterStatus;
        $filterDate = explode('-', $request->filterDate);
        $startDate = $filterDate[0];
        $endDate = isset($filterDate[1]) ? trim($filterDate[1]) : null;

        $singleSends = SingleSend::with('user')
            ->whereHas('user')
            ->orderBy('id', 'DESC')
            ->when(isset($filterName), function ($query) use ($filterName) {
                return $query->where('single_send_name', 'LIKE', '%' . $filterName . '%');
            })
            ->when(isset($filterCreator), function ($query) use ($filterCreator) {
                $query->whereHas('user', function ($qq) use ($filterCreator) {
                    $qq->where('firstname', 'LIKE', '%' . $filterCreator . '%')
                        ->Orwhere('lastname', 'LIKE', '%' . $filterCreator . '%')
                        ->Orwhere('username', 'LIKE', '%' . $filterCreator . '%');
                });
            })
            ->when(isset($filterStatus), function ($query) use ($filterStatus) {
                if ($filterStatus != "all") {
                    if ($filterStatus == 'schedule') {
                        return $query->where('schedule_type', 1);
                    } else {
                        return $query->where('status', $filterStatus);
                    }
                }
            })
            ->when(!empty($request->filterDate) && $endDate == null, function ($query) use ($startDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $query->whereDate('created_at', $startDate);
            })
            ->when(!empty($request->filterDate) && $endDate != null, function ($query) use ($startDate, $endDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $endDate = Carbon::createFromFormat('d/m/Y', trim($endDate));
                $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->when(!empty($request->search['value']), function ($query) use ($request) {
                $query->where(function ($subquery) use ($request) {
                    $subquery->where('single_send_name', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhereHas('user', function ($userQuery) use ($request) {
                            $userQuery->where('firstname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('lastname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('username', 'LIKE', '%' . $request->search['value'] . '%');
                        });
                });
            });


        return DataTables::of($singleSends)
            ->addColumn('checkbox', function ($item) {
                return '<input type="checkbox" id="chk-' . $item->id . '"
                                       class="form-check-input row-tic tic-check" name="check" value="' . $item->id . '"
                                       data-id="' . $item->id . '">';
            })
            ->addColumn('no', function () {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('singleSendName', function ($item) {
                if ($item->status == 3) {
                    return ' <div><span class="legend-indicator bg-success"></span>' . trans('Sent on ') . dateTime($item->sent_at, basicControl()->date_time_format) . '</div> <strong>' . $item->single_send_name . '</strong>';
                } elseif ($item->status == 2) {
                    return ' <div><span class="legend-indicator bg-warning"></span>' . trans('Running Sent on ') . dateTime($item->sent_at, basicControl()->date_time_format) . '</div> <strong>' . $item->single_send_name . '</strong>';
                } elseif ($item->status == 1 && $item->schedule_type == 1) {
                    return ' <div><span class="legend-indicator bg-dark"></span>' . trans('Schedule on ') . dateTime($item->schedule_date, 'd M Y') . ' ' . $item->schedule_time . '</div> <strong>' . $item->single_send_name . '</strong>';
                } else {
                    return '<div><span class="legend-indicator bg-primary"></span>' . trans('Draft') . '</div> <strong>' . $item->single_send_name . '</strong>';
                }
            })
            ->addColumn('delivered', function ($item) {
                if (isset($item->delivered)) {
                    return ' <div class="progress" style="height: 15px;">
                             <div class="progress-bar bg-success" role="progressbar"
                             style="width: ' . $item->getPercent()['deliveredPercent'] . '%;"
                             aria-valuenow="25" aria-valuemin="0"
                             aria-valuemax="100">' . $item->getPercent()['deliveredPercent'] . '%
                             </div></div> ';
                } else {
                    return '-';
                }
            })
            ->addColumn('bounces', function ($item) {
                if (isset($item->bounces)) {
                    return ' <div class="progress" style="height: 15px;">
                             <div class="progress-bar bg-danger" role="progressbar"
                             style="width: ' . $item->getPercent()['bouncesPercent'] . '%;"
                             aria-valuenow="25" aria-valuemin="0"
                             aria-valuemax="100">' . $item->getPercent()['bouncesPercent'] . '%
                             </div></div> ';
                } else {
                    return '-';
                }
            })
            ->addColumn('uniqueOpens', function ($item) {
                if (isset($item->unique_opens)) {
                    return ' <div class="progress" style="height: 15px;">
                             <div class="progress-bar bg-primary" role="progressbar"
                             style="width: ' . $item->getPercent()['uniqueOpenPercent'] . '%;"
                             aria-valuenow="25" aria-valuemin="0"
                             aria-valuemax="100">' . $item->getPercent()['uniqueOpenPercent'] . '%
                             </div></div> ';
                } else {
                    return '-';
                }
            })
            ->addColumn('unsubscribe', function ($item) {
                if (isset($item->unsubscribes)) {
                    return ' <div class="progress" style="height: 15px;">
                             <div class="progress-bar bg-warning" role="progressbar"
                             style="width: ' . $item->getPercent()['unsubscribesPercent'] . '%;"
                             aria-valuenow="25" aria-valuemin="0"
                             aria-valuemax="100">' . $item->getPercent()['unsubscribesPercent'] . '%
                             </div></div> ';
                } else {
                    return '-';
                }
            })
            ->addColumn('creator', function ($item) {
                $userProfileRoute = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="' . $userProfileRoute . '">
                                <div class="flex-shrink-0">
                                  ' . optional($item->user)->profilePicture() . '
                                </div>
                                <div class="flex-grow-1 ms-3">
                                  <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                                  <span class="fs-6 text-body">' . optional($item->user)->username . '</span>
                                </div>
                              </a>';
            })
            ->addColumn('created_at', function ($item) {
                return dateTime($item->created_at, basicControl()->date_time_format);
            })
            ->addColumn('action', function ($item) {
                $viewStatics = route('admin.singleSendStats', $item->utr);
                $delete = route('admin.singleSendDelete', $item->utr);
                return '<div class="btn-group" role="group">
                      <a href="' . $viewStatics . '" class="btn btn-white btn-sm edit_user_btn">
                        <i class="fal fa-line-chart me-1"></i> ' . trans("View Statistics") . '
                      </a>
                    <div class="btn-group">
                      <button type="button" class="btn btn-white btn-icon btn-sm dropdown-toggle dropdown-toggle-empty" id="userEditDropdown" data-bs-toggle="dropdown" aria-expanded="false"></button>
                      <div class="dropdown-menu dropdown-menu-end mt-1" aria-labelledby="userEditDropdown">
                        <a class="dropdown-item delete_btn" href="javascript:void(0)" data-bs-target="#delete"
                           data-bs-toggle="modal" data-route="' . $delete . '">
                          <i class="fal fa-trash dropdown-item-icon"></i> ' . trans("Delete") . '
                       </a>
                      </div>
                    </div>
                  </div>';
            })
            ->rawColumns(['checkbox', 'singleSendName', 'delivered', 'bounces', 'uniqueOpens', 'unsubscribe', 'creator', 'created_at', 'action'])
            ->make(true);
    }

    public function singleSendStats($utr)
    {
        $singleSend = SingleSend::with(['sender', 'template', 'unsubscribeGroup', 'singleSendActivities'])
            ->withCount(['singleSendActivitiesUnsubscribes'])->where('utr', $utr)->firstOrFail();
        $singleSend->unsubscribes = $singleSend->single_send_activities_unsubscribes_count;
        $singleSend->save();


        $data['deliveredPercent'] = $singleSend->getPercent()['deliveredPercent'];
        $data['bouncesPercent'] = $singleSend->getPercent()['bouncesPercent'];
        $data['uniqueOpenPercent'] = $singleSend->getPercent()['uniqueOpenPercent'];
        $data['unsubscribesPercent'] = $singleSend->getPercent()['unsubscribesPercent'];

        $data['horizontalBarChat'] = [$singleSend->triggered, $singleSend->delivered, $singleSend->bounces,
            $singleSend->unique_opens, $singleSend->total_opens, $singleSend->unsubscribes];

        $data['lineChartUniqueOpen'] = DB::table('single_send_activities')
            ->select(DB::raw('triggered_date AS x'), DB::raw('COUNT(*) AS y'))
            ->where('single_send_id', $singleSend->id)
            ->where('type', '3')
            ->groupBy('triggered_date')
            ->get()
            ->toArray();


        $start = Carbon::now()->firstOfMonth();
        $end = Carbon::now()->lastOfMonth();

        $transactions = SingleSendActivity::select(DB::raw("DATE_FORMAT(triggered_date, '%j') as day"))
            ->whereBetween('triggered_date', [$start, $end])
            ->selectRaw("COUNT(CASE WHEN type = '3' THEN 1 END) as `Unique`")
            ->selectRaw("COUNT(CASE WHEN type = '4' THEN 1 END) as `Unsubscribe`")
            ->selectRaw("triggered_date as `triggered_date`")
            ->where('single_send_id', $singleSend->id)
            ->groupBy('day')
            ->get();

        $labels = [];
        $dataUnique = [];
        $dataUnsubscribe = [];
        $start = new DateTime($start);
        $end = new DateTime($end);

        for ($day = $start; $day <= $end; $day->modify('+1 day')) {
            $i = $day->format('j');
            $labels[] = $day->format('jS M');
            $currentUnique = 0;
            $currentUnsubscribe = 0;

            if (isset($transactions)) {
                foreach ($transactions as $key => $transaction) {
                    if (Carbon::parse($transaction->triggered_date) == $day) {
                        $currentUnique = $transaction->Unique;
                        $currentUnsubscribe = $transaction->Unsubscribe;
                        break;
                    }
                }
            }

            $dataUnique[] = $currentUnique;
            $dataUnsubscribe[] = $currentUnsubscribe;
        }

        $data['labels'] = $labels;
        $data['dataUnique'] = $dataUnique;
        $data['dataUnsubscribe'] = $dataUnsubscribe;
        $data['singleSend'] = $singleSend;
        return view('admin.single-send.stats', $data);
    }

    public function singleSendGetChart(Request $request, $utr)
    {
        $singleSend = SingleSend::select(['id', 'utr'])->where('utr', $utr)->first();
        $start = $request->start;
        $end = $request->end;

        $transactions = SingleSendActivity::select(DB::raw("DATE_FORMAT(triggered_date, '%j') as day"))
            ->whereBetween('triggered_date', [$start, $end])
            ->selectRaw("COUNT(CASE WHEN type = '3' THEN 1 END) as `Unique`")
            ->selectRaw("COUNT(CASE WHEN type = '4' THEN 1 END) as `Unsubscribe`")
            ->selectRaw("triggered_date as `triggered_date`")
            ->where('single_send_id', $singleSend->id)
            ->groupBy('day')
            ->get();

        $labels = [];
        $dataUnique = [];
        $dataUnsubscribe = [];
        $start = new DateTime($start);
        $end = new DateTime($end);

        for ($day = $start; $day <= $end; $day->modify('+1 day')) {
            $i = $day->format('j');
            $labels[] = $day->format('jS M');
            $currentUnique = 0;
            $currentUnsubscribe = 0;

            if (isset($transactions)) {
                foreach ($transactions as $key => $transaction) {
                    if (Carbon::parse($transaction->triggered_date) == $day) {
                        $currentUnique = $transaction->Unique;
                        $currentUnsubscribe = $transaction->Unsubscribe;
                        break;
                    }
                }
            }

            $dataUnique[] = $currentUnique;
            $dataUnsubscribe[] = $currentUnsubscribe;
        }

        $data['labels'] = $labels;
        $data['dataUnique'] = $dataUnique;
        $data['dataUnsubscribe'] = $dataUnsubscribe;

        return response()->json($data);
    }

    public function singleSendDelete($utr)
    {
        $singleSend = SingleSend::where('status', '!=', 2)->where('utr', $utr)->firstOrFail();
        $singleSend->delete();

        return back()->with('success', 'Deleted Successfully');
    }

    public function singleSendDeleteMultiple(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select User.');
            return response()->json(['error' => 1]);
        } else {
            SingleSend::whereIn('id', $request->strIds)->delete();
            session()->flash('success', 'Single Send has been deleted successfully');
            return response()->json(['success' => 1]);
        }
    }
}
