<?php

namespace App\Http\Controllers\Admin\GeneratedHistory;

use App\Http\Controllers\Controller;
use App\Models\CustomField;
use Illuminate\Http\Request;
use Yajra\DataTables\Facades\DataTables;

class CustomFieldController extends Controller
{
    public function index()
    {
        $data['fieldRecord'] = CustomField::selectRaw('COUNT(id) AS totalField')
            ->selectRaw('COUNT(CASE WHEN is_reserved = 1 THEN id END) AS reserveField')
            ->selectRaw('(COUNT(CASE WHEN is_reserved = 1 THEN id END) / COUNT(id)) * 100 AS reserveFieldPercentage')
            ->selectRaw('COUNT(CASE WHEN is_reserved = 0 AND field_type = "text" THEN id END) AS customTextField')
            ->selectRaw('(COUNT(CASE WHEN is_reserved = 0 AND field_type = "text" THEN id END) / COUNT(id)) * 100 AS customTextFieldPercentage')
            ->selectRaw('COUNT(CASE WHEN is_reserved = 0 AND field_type = "number" THEN id END) AS customNumberField')
            ->selectRaw('(COUNT(CASE WHEN is_reserved = 0 AND field_type = "number" THEN id END) / COUNT(id)) * 100 AS customNumberFieldPercentage')
            ->selectRaw('COUNT(CASE WHEN is_reserved = 0 AND field_type = "datetime-local" THEN id END) AS customDateField')
            ->selectRaw('(COUNT(CASE WHEN is_reserved = 0 AND field_type = "datetime-local" THEN id END) / COUNT(id)) * 100 AS customDateFieldPercentage')
            ->get()
            ->toArray();

        $data['reserveFields'] = CustomField::where('is_reserved', 1)->get();

        return view('admin.custom-field.list', $data);
    }

    public function customFieldSearch(Request $request)
    {
        $customFields = CustomField::with('user:id,firstname,lastname,username,image,image_driver')
            ->whereHas('user')
            ->when(!empty($request->search['value']), function ($query) use ($request) {
                $query->where(function ($subquery) use ($request) {
                    $subquery->where('field_name', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('field_type', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhereHas('user', function ($userQuery) use ($request) {
                            $userQuery->where('firstname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('lastname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('username', 'LIKE', '%' . $request->search['value'] . '%');
                        });
                });
            })
            ->orderBy('id', 'DESC')->whereNotNull('user_id');


        return DataTables::of($customFields)
            ->addColumn('checkbox', function ($item) {
                return '<input type="checkbox" id="chk-' . $item->id . '"
                                       class="form-check-input row-tic tic-check" name="check" value="' . $item->id . '"
                                       data-id="' . $item->id . '">';
            })
            ->addColumn('no', function () {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('name', function ($item) {
                return $item->field_name;
            })
            ->addColumn('type', function ($item) {
                return $item->field_type;
            })
            ->addColumn('creator', function ($item) {
                $userProfileRoute = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="' . $userProfileRoute . '">
                                <div class="flex-shrink-0">
                                  ' . optional($item->user)->profilePicture() . '
                                </div>
                                <div class="flex-grow-1 ms-3">
                                  <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                                  <span class="fs-6 text-body">' . optional($item->user)->username . '</span>
                                </div>
                              </a>';
            })
            ->addColumn('action', function ($item) {
                $delete = route('admin.customFieldDelete', $item->id);
                return '<div class="btn-group" role="group">
                      <a href="javascript:void(0)" class="btn btn-white btn-sm delete_btn" data-bs-target="#delete"
                           data-bs-toggle="modal" data-route="' . $delete . '">
                        <i class="fal fa-trash me-1"></i> ' . trans("Delete") . '
                      </a>
                  </div>';
            })
            ->rawColumns(['checkbox', 'name', 'type', 'creator', 'action'])
            ->make(true);
    }

    public function customFieldDelete($id)
    {
        $customField = CustomField::findOrFail($id);
        $customField->delete();
        return back()->with('success', 'Custom Field Deleted Successfully');
    }

    public function customFieldMultipleDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select User.');
            return response()->json(['error' => 1]);
        } else {
            CustomField::whereIn('id', $request->strIds)->delete();
            session()->flash('success', 'Custom Field has been deleted successfully');
            return response()->json(['success' => 1]);
        }
    }
}
