<?php

namespace App\Http\Controllers\Admin\GeneratedHistory;

use App\Http\Controllers\Controller;
use App\Models\Contact;
use App\Models\ContactType;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Str;
use Yajra\DataTables\Facades\DataTables;

class ContactController extends Controller
{
    public function contactGroup()
    {
        return view('admin.contact.group');
    }

    public function contactGroupSearch(Request $request)
    {
        $contactTypes = ContactType::with('user:id,firstname,lastname,username,image,image_driver')
            ->whereHas('user')
            ->withCount('contacts')->where('name', '!=', 'Global')
            ->when(!empty($request->search['value']), function ($query) use ($request) {
                $query->where(function ($subquery) use ($request) {
                    $subquery->where('name', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhereHas('user', function ($userQuery) use ($request) {
                            $userQuery->where('firstname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('lastname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('username', 'LIKE', '%' . $request->search['value'] . '%');
                        });
                });
            })->orderBy('id', 'DESC');


        return DataTables::of($contactTypes)
            ->addColumn('checkbox', function ($item) {
                return '<input type="checkbox" id="chk-' . $item->id . '"
                                       class="form-check-input row-tic tic-check" name="check" value="' . $item->id . '"
                                       data-id="' . $item->id . '">';
            })
            ->addColumn('no', function () {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('name', function ($item) {
                return $item->name;
            })
            ->addColumn('activeContact', function ($item) {
                return '<span class="badge bg-soft-dark text-dark">' . $item->contacts_count . '</span>';
            })
            ->addColumn('creator', function ($item) {
                $userProfileRoute = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="' . $userProfileRoute . '">
                                <div class="flex-shrink-0">
                                  ' . optional($item->user)->profilePicture() . '
                                </div>
                                <div class="flex-grow-1 ms-3">
                                  <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                                  <span class="fs-6 text-body">' . optional($item->user)->username . '</span>
                                </div>
                              </a>';
            })
            ->addColumn('created_at', function ($item) {
                return dateTime($item->created_at, basicControl()->date_time_format);
            })
            ->addColumn('action', function ($item) {
                $showUrl = route('admin.contactList') . '?contact_type=' . $item->id;
                $delete = route('admin.contactTypeDelete', $item->id);
                return '<div class="btn-group" role="group">
                      <a href="' . $showUrl . '" class="btn btn-white btn-sm edit_user_btn">
                        <i class="fal fa-users me-1"></i> ' . trans("Contact List") . '
                      </a>
                    <div class="btn-group">
                      <button type="button" class="btn btn-white btn-icon btn-sm dropdown-toggle dropdown-toggle-empty" id="userEditDropdown" data-bs-toggle="dropdown" aria-expanded="false"></button>
                      <div class="dropdown-menu dropdown-menu-end mt-1" aria-labelledby="userEditDropdown">
                        <a class="dropdown-item delete_btn" href="javascript:void(0)" data-bs-target="#delete"
                           data-bs-toggle="modal" data-route="' . $delete . '">
                          <i class="fal fa-trash dropdown-item-icon"></i> ' . trans("Delete") . '
                       </a>
                      </div>
                    </div>
                  </div>';
            })
            ->rawColumns(['checkbox', 'name', 'activeContact', 'creator', 'created_at', 'action'])
            ->make(true);
    }

    public function contactTypeDelete($id)
    {
        $contactType = ContactType::findOrFail($id);
        $contactType->delete();
        return back()->with('success', 'Deleted Successfully');
    }

    public function contactList(Request $request)
    {
        $data['contactType'] = $request->contact_type ?? null;
        $data['contactRecord'] = Contact::selectRaw('COUNT(id) AS totalContact')
            ->selectRaw('COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) AS todayContact')
            ->selectRaw('(COUNT(CASE WHEN DATE(created_at) = CURRENT_DATE THEN id END) / COUNT(id)) * 100 AS todayContactPercentage')
            ->selectRaw('COUNT(CASE WHEN YEARWEEK(created_at, 1) = YEARWEEK(CURDATE(), 1) THEN id END) AS thisWeekContact')
            ->selectRaw('(COUNT(CASE WHEN YEARWEEK(created_at, 1) = YEARWEEK(CURDATE(), 1) THEN id END) / COUNT(id)) * 100 AS thisWeekContactPercentage')
            ->selectRaw('COUNT(CASE WHEN MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE()) THEN id END) AS thisMonthContact')
            ->selectRaw('(COUNT(CASE WHEN MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE()) THEN id END) / COUNT(id)) * 100 AS thisMonthContactPercentage')
            ->selectRaw('COUNT(CASE WHEN deleted_at IS NOT NULL THEN id END) AS deletedContact')
            ->selectRaw('(COUNT(CASE WHEN deleted_at IS NOT NULL THEN id END) / COUNT(id)) * 100 AS deletedContactPercentage')
            ->get()
            ->toArray();

        $data['contactDeletedRecord'] = Contact::withTrashed()->selectRaw('COUNT(id) AS totalDeletedContact')
            ->selectRaw('COUNT(CASE WHEN deleted_at IS NOT NULL THEN id END) AS deletedContact')
            ->selectRaw('(COUNT(CASE WHEN deleted_at IS NOT NULL THEN id END) / COUNT(id)) * 100 AS deletedContactPercentage')
            ->get()
            ->toArray();
        return view('admin.contact.list', $data);
    }

    public function contactListSearch(Request $request)
    {
        $contactType = $request->contactType ?? null;
        $filterEmail = $request->email;
        $filterCreator = $request->creator;
        $filterDate = explode('-', $request->filterDate);
        $startDate = $filterDate[0];
        $endDate = isset($filterDate[1]) ? trim($filterDate[1]) : null;

        $contacts = Contact::with(['user:id,firstname,lastname,username,image,image_driver'])
            ->whereHas('user')
            ->orderBy('id', 'DESC')
            ->when($contactType, function ($query) use ($contactType) {
                return $query->where('contact_type_id', $contactType);
            })
            ->when(isset($filterEmail), function ($query) use ($filterEmail) {
                return $query->where('email', 'LIKE', '%' . $filterEmail . '%');
            })
            ->when(isset($filterCreator), function ($query) use ($filterCreator) {
                $query->whereHas('user', function ($qq) use ($filterCreator) {
                    $qq->where('firstname', 'LIKE', '%' . $filterCreator . '%')
                        ->orWhere('lastname', 'LIKE', '%' . $filterCreator . '%')
                        ->orWhere('username', 'LIKE', '%' . $filterCreator . '%');
                });
            })
            ->when(!empty($request->filterDate) && $endDate == null, function ($query) use ($startDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $query->whereDate('created_at', $startDate);
            })
            ->when(!empty($request->filterDate) && $endDate != null, function ($query) use ($startDate, $endDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $endDate = Carbon::createFromFormat('d/m/Y', trim($endDate));
                $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->when(!empty($request->search['value']), function ($query) use ($request) {
                $query->where(function ($subquery) use ($request) {
                    $subquery->where('email', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('first_name', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhere('last_name', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhereHas('user', function ($qq) use ($request) {
                            $qq->where('firstname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('lastname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('username', 'LIKE', '%' . $request->search['value'] . '%');
                        });
                });
            });

        return DataTables::of($contacts)
            ->addColumn('checkbox', function ($item) {
                return '<input type="checkbox" id="chk-' . $item->id . '"
                                       class="form-check-input row-tic tic-check" name="check" value="' . $item->id . '"
                                       data-id="' . $item->id . '">';
            })
            ->addColumn('no', function () {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('name', function ($item) {
                return $item->fullname;
            })
            ->addColumn('email', function ($item) {
                return $item->email;
            })
            ->addColumn('group', function ($item) {
                return optional($item->contactType)->name ?? '-';
            })
            ->addColumn('creator', function ($item) {
                $userProfileRoute = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="' . $userProfileRoute . '">
                                <div class="flex-shrink-0">
                                  ' . optional($item->user)->profilePicture() . '
                                </div>
                                <div class="flex-grow-1 ms-3">
                                  <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                                  <span class="fs-6 text-body">' . optional($item->user)->username . '</span>
                                </div>
                              </a>';
            })
            ->addColumn('created_at', function ($item) {
                return dateTime($item->created_at, basicControl()->date_time_format);
            })
            ->addColumn('action', function ($item) {
                $delete = route('admin.contactListDelete', $item->id);
                return '<div class="btn-group" role="group">
                      <a href="javascript:void(0)" data-bs-target="#delete" data-bs-toggle="modal"
                       data-route="' . $delete . '" class="btn btn-white btn-sm delete_btn">
                        <i class="fal fa-trash me-1"></i> ' . trans("Delete") . '
                      </a>
                  </div>';
            })
            ->rawColumns(['checkbox', 'name', 'email', 'group', 'creator', 'created_at', 'action'])
            ->make(true);
    }

    public function contactListDelete($id)
    {
        $contact = Contact::findOrFail($id);
        $contact->delete();
        return back()->with('success', 'Deleted Successfully');
    }

    public function contactTypeMultiDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select User.');
            return response()->json(['error' => 1]);
        } else {
            ContactType::whereIn('id', $request->strIds)->delete();
            session()->flash('success', 'Contact Type has been deleted successfully');
            return response()->json(['success' => 1]);
        }
    }

    public function contactListMultiDelete(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select User.');
            return response()->json(['error' => 1]);
        } else {
            Contact::whereIn('id', $request->strIds)->delete();
            session()->flash('success', 'Contact has been deleted successfully');
            return response()->json(['success' => 1]);
        }
    }
}
