<?php

namespace App\Http\Controllers\Admin\GeneratedHistory;

use App\Http\Controllers\Controller;
use App\Models\Automation;
use App\Models\EmailAutomationMap;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;
use DateTime;

class AutomationController extends Controller
{
    public function index()
    {
        $automationRecord = Automation::selectRaw('COUNT(id) AS totalAutomation')
            ->selectRaw('COUNT(CASE WHEN status = 1 THEN id END) AS liveAutomation')
            ->selectRaw('(COUNT(CASE WHEN status = 1 THEN id END) / COUNT(id)) * 100 AS liveAutomationPercentage')
            ->selectRaw('COUNT(CASE WHEN status = 0 THEN id END) AS draftAutomation')
            ->selectRaw('(COUNT(CASE WHEN status = 0 THEN id END) / COUNT(id)) * 100 AS draftAutomationPercentage')
            ->selectRaw('COUNT(CASE WHEN automation_type = "welcome" THEN id END) AS welcomeAutomation')
            ->selectRaw('(COUNT(CASE WHEN automation_type = "welcome" THEN id END) / COUNT(id)) * 100 AS welcomeAutomationPercentage')
            ->selectRaw('COUNT(CASE WHEN automation_type = "custom" THEN id END) AS customAutomation')
            ->selectRaw('(COUNT(CASE WHEN automation_type = "custom" THEN id END) / COUNT(id)) * 100 AS customAutomationPercentage')
            ->get()
            ->toArray();

        return view('admin.automation.list', compact('automationRecord'));
    }

    public function automationSearch(Request $request)
    {
        $filterName = $request->name;
        $filterCreator = $request->creator;
        $filterStatus = $request->filterStatus;
        $filterDate = explode('-', $request->filterDate);
        $startDate = $filterDate[0];
        $endDate = isset($filterDate[1]) ? trim($filterDate[1]) : null;

        $automations = Automation::with('user')->withCount(['automationSchedule'])
            ->whereHas('user')
            ->orderBy('id', 'DESC')
            ->when(isset($filterName), function ($query) use ($filterName) {
                return $query->where('automation_name', 'LIKE', '%' . $filterName . '%');
            })
            ->when(isset($filterCreator), function ($query) use ($filterCreator) {
                $query->whereHas('user', function ($qq) use ($filterCreator) {
                    $qq->where('firstname', 'LIKE', '%' . $filterCreator . '%')
                        ->orWhere('lastname', 'LIKE', '%' . $filterCreator . '%')
                        ->orWhere('username', 'LIKE', '%' . $filterCreator . '%');
                });
            })
            ->when(isset($filterStatus), function ($query) use ($filterStatus) {
                if ($filterStatus != "all") {
                    return $query->where('status', $filterStatus);
                }
            })
            ->when(!empty($request->filterDate) && $endDate == null, function ($query) use ($startDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $query->whereDate('created_at', $startDate);
            })
            ->when(!empty($request->filterDate) && $endDate != null, function ($query) use ($startDate, $endDate) {
                $startDate = Carbon::createFromFormat('d/m/Y', trim($startDate));
                $endDate = Carbon::createFromFormat('d/m/Y', trim($endDate));
                $query->whereBetween('created_at', [$startDate, $endDate]);
            })
            ->when(!empty($request->search['value']), function ($query) use ($request) {
                $query->where(function ($subquery) use ($request) {
                    $subquery->where('automation_name', 'LIKE', '%' . $request->search['value'] . '%')
                        ->orWhereHas('user', function ($userQuery) use ($request) {
                            $userQuery->where('firstname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('lastname', 'LIKE', '%' . $request->search['value'] . '%')
                                ->orWhere('username', 'LIKE', '%' . $request->search['value'] . '%');
                        });
                });
            });


        return DataTables::of($automations)
            ->addColumn('checkbox', function ($item) {
                return '<input type="checkbox" id="chk-' . $item->id . '"
                                       class="form-check-input row-tic tic-check" name="check" value="' . $item->id . '"
                                       data-id="' . $item->id . '">';
            })
            ->addColumn('no', function () {
                static $counter = 0;
                $counter++;
                return $counter;
            })
            ->addColumn('automationName', function ($item) {
                if ($item->status) {
                    return ' <div><span class="legend-indicator bg-warning"></span>' . trans('Live On ') .
                        dateTime($item->live_at, basicControl()->date_time_format) . ' -<i class="fal fa-mail-bulk"></i> ' . $item->automation_schedule_count . '
                              </div> <strong>' . $item->automation_name . '</strong>';
                } else {
                    return ' <div><span class="legend-indicator bg-primary"></span>' . trans('Draft - ') .
                        '<i class="fal fa-mail-bulk"></i> ' . $item->automation_schedule_count . '</div> <strong>' . $item->automation_name . '</strong>';
                }
            })
            ->addColumn('delivered', function ($item) {
                if ($item->total_triggered) {
                    return ' <div class="progress" style="height: 15px;">
                             <div class="progress-bar bg-success" role="progressbar"
                             style="width: ' . $item->getPercent()['deliveredPercent'] . '%;"
                             aria-valuenow="25" aria-valuemin="0"
                             aria-valuemax="100">' . $item->getPercent()['deliveredPercent'] . '%
                             </div></div> ';
                } else {
                    return '-';
                }
            })
            ->addColumn('bounces', function ($item) {
                if ($item->total_triggered) {
                    return ' <div class="progress" style="height: 15px;">
                             <div class="progress-bar bg-danger" role="progressbar"
                             style="width: ' . $item->getPercent()['bouncesPercent'] . '%;"
                             aria-valuenow="25" aria-valuemin="0"
                             aria-valuemax="100">' . $item->getPercent()['bouncesPercent'] . '%
                             </div></div> ';
                } else {
                    return '-';
                }
            })
            ->addColumn('uniqueOpens', function ($item) {
                if ($item->total_triggered) {
                    return ' <div class="progress" style="height: 15px;">
                             <div class="progress-bar bg-primary" role="progressbar"
                             style="width: ' . $item->getPercent()['uniqueOpenPercent'] . '%;"
                             aria-valuenow="25" aria-valuemin="0"
                             aria-valuemax="100">' . $item->getPercent()['uniqueOpenPercent'] . '%
                             </div></div> ';
                } else {
                    return '-';
                }
            })
            ->addColumn('unsubscribe', function ($item) {
                if ($item->total_triggered) {
                    return ' <div class="progress" style="height: 15px;">
                             <div class="progress-bar bg-warning" role="progressbar"
                             style="width: ' . $item->getPercent()['unsubscribesPercent'] . '%;"
                             aria-valuenow="25" aria-valuemin="0"
                             aria-valuemax="100">' . $item->getPercent()['unsubscribesPercent'] . '%
                             </div></div> ';
                } else {
                    return '-';
                }
            })
            ->addColumn('creator', function ($item) {
                $userProfileRoute = route('admin.user.view.profile', $item->user_id);
                return '<a class="d-flex align-items-center me-2" href="' . $userProfileRoute . '">
                                <div class="flex-shrink-0">
                                  ' . optional($item->user)->profilePicture() . '
                                </div>
                                <div class="flex-grow-1 ms-3">
                                  <h5 class="text-hover-primary mb-0">' . optional($item->user)->firstname . ' ' . optional($item->user)->lastname . '</h5>
                                  <span class="fs-6 text-body">' . optional($item->user)->username . '</span>
                                </div>
                              </a>';
            })
            ->addColumn('created_at', function ($item) {
                return dateTime($item->created_at, basicControl()->date_time_format);
            })
            ->addColumn('action', function ($item) {
                $showUrl = route('admin.automation.show', $item->utr);
                $viewStatics = route('admin.automationStatics', $item->utr);
                $delete = route('admin.automationDelete', $item->utr);
                return '<div class="btn-group" role="group">
                      <a href="' . $showUrl . '" class="btn btn-white btn-sm edit_user_btn">
                        <i class="fal fa-eye me-1"></i> ' . trans("View") . '
                      </a>
                    <div class="btn-group">
                      <button type="button" class="btn btn-white btn-icon btn-sm dropdown-toggle dropdown-toggle-empty" id="userEditDropdown" data-bs-toggle="dropdown" aria-expanded="false"></button>
                      <div class="dropdown-menu dropdown-menu-end mt-1" aria-labelledby="userEditDropdown">
                       <a class="dropdown-item" href="' . $viewStatics . '">
                          <i class="fal fa-line-chart dropdown-item-icon"></i> ' . trans("View Statistics") . '
                       </a>
                        <a class="dropdown-item delete_btn" href="javascript:void(0)" data-bs-target="#delete"
                           data-bs-toggle="modal" data-route="' . $delete . '">
                          <i class="fal fa-trash dropdown-item-icon"></i> ' . trans("Delete") . '
                       </a>
                      </div>
                    </div>
                  </div>';
            })
            ->rawColumns(['checkbox', 'automationName', 'delivered', 'bounces', 'uniqueOpens', 'unsubscribe', 'creator', 'created_at', 'action'])
            ->make(true);
    }

    public function show($utr)
    {
        $data['automation'] = Automation::with(['automationSchedule:id,user_id,automation_id,sender_id,subject,waiting_period,waiting_type,utr,template_id,preview_img'])
            ->where('utr', $utr)->firstOrFail();
        return view('admin.automation.show', $data);
    }

    public function automationStatics($utr)
    {
        $automation = Automation::with(['automationSchedule', 'emailAutomationMaps', 'unsubscribeGroup',
            'contactType', 'segment'])
            ->where('utr', $utr)->firstOrFail();


        $data['deliveredPercent'] = $automation->getPercent()['deliveredPercent'];
        $data['bouncesPercent'] = $automation->getPercent()['bouncesPercent'];
        $data['uniqueOpenPercent'] = $automation->getPercent()['uniqueOpenPercent'];
        $data['unsubscribesPercent'] = $automation->getPercent()['unsubscribesPercent'];


        $data['horizontalBarChat'] = [$automation->total_triggered, $automation->total_delivered, $automation->total_bounces,
            $automation->total_unique_opens, $automation->total_opens, $automation->total_unsubscribe];

        $data['lineChartUniqueOpen'] = DB::table('email_automation_maps')
            ->select(DB::raw('open_date AS x'), DB::raw('COUNT(*) AS y'))
            ->where('automation_id', $automation->id)
            ->where('unique_opens', 1)
            ->groupBy('open_date')
            ->get()
            ->toArray();


        $start = Carbon::now()->firstOfMonth();
        $end = Carbon::now()->lastOfMonth();

        $transactions = EmailAutomationMap::select(DB::raw("DATE_FORMAT(open_date, '%j') as day"))
            ->whereBetween('open_date', [$start, $end])
            ->selectRaw("COUNT(CASE WHEN unique_opens = '1' THEN 1 END) as `Unique`")
            ->selectRaw("COUNT(CASE WHEN unsubscribes = '1' THEN 1 END) as `Unsubscribe`")
            ->selectRaw("open_date as `open_date`")
            ->where('automation_id', $automation->id)
            ->groupBy('day')
            ->get();

        $labels = [];
        $dataUnique = [];
        $dataUnsubscribe = [];
        $start = new DateTime($start);
        $end = new DateTime($end);

        for ($day = $start; $day <= $end; $day->modify('+1 day')) {
            $i = $day->format('j');
            $labels[] = $day->format('jS M');
            $currentUnique = 0;
            $currentUnsubscribe = 0;

            if (isset($transactions)) {
                foreach ($transactions as $key => $transaction) {
                    if (Carbon::parse($transaction->open_date) == $day) {
                        $currentUnique = $transaction->Unique;
                        $currentUnsubscribe = $transaction->Unsubscribe;
                        break;
                    }
                }
            }

            $dataUnique[] = $currentUnique;
            $dataUnsubscribe[] = $currentUnsubscribe;
        }

        $data['labels'] = $labels;
        $data['dataUnique'] = $dataUnique;
        $data['dataUnsubscribe'] = $dataUnsubscribe;
        $data['automation'] = $automation;
        return view('admin.automation.stats', $data);
    }

    public function automationsGetChart(Request $request, $utr)
    {
        $automation = Automation::select(['id', 'utr'])->where('utr', $utr)->first();
        $start = $request->start;
        $end = $request->end;

        $transactions = EmailAutomationMap::select(DB::raw("DATE_FORMAT(open_date, '%j') as day"))
            ->whereBetween('open_date', [$start, $end])
            ->selectRaw("COUNT(CASE WHEN unique_opens = '1' THEN 1 END) as `Unique`")
            ->selectRaw("COUNT(CASE WHEN unsubscribes = '1' THEN 1 END) as `Unsubscribe`")
            ->selectRaw("open_date as `open_date`")
            ->where('automation_id', $automation->id)
            ->groupBy('day')
            ->get();

        $labels = [];
        $dataUnique = [];
        $dataUnsubscribe = [];
        $start = new DateTime($start);
        $end = new DateTime($end);

        for ($day = $start; $day <= $end; $day->modify('+1 day')) {
            $i = $day->format('j');
            $labels[] = $day->format('jS M');
            $currentUnique = 0;
            $currentUnsubscribe = 0;

            if (isset($transactions)) {
                foreach ($transactions as $key => $transaction) {
                    if (Carbon::parse($transaction->open_date) == $day) {
                        $currentUnique = $transaction->Unique;
                        $currentUnsubscribe = $transaction->Unsubscribe;
                        break;
                    }
                }
            }

            $dataUnique[] = $currentUnique;
            $dataUnsubscribe[] = $currentUnsubscribe;
        }

        $data['labels'] = $labels;
        $data['dataUnique'] = $dataUnique;
        $data['dataUnsubscribe'] = $dataUnsubscribe;

        return response()->json($data);
    }

    public function automationDelete($utr)
    {
        Automation::where('utr', $utr)->firstOrfail()->delete();
        return back()->with('success', 'Deleted Successfully');
    }

    public function automationDeleteMultiple(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select User.');
            return response()->json(['error' => 1]);
        } else {
            Automation::whereIn('id', $request->strIds)->delete();
            session()->flash('success', 'Automation has been deleted successfully');
            return response()->json(['success' => 1]);
        }
    }
}
