<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Country;
use App\Traits\Upload;
use Illuminate\Http\Request;
use Illuminate\Support\Arr;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Validator;
use Yajra\DataTables\Facades\DataTables;

class CountryController extends Controller
{
    use Upload;

    public function index()
    {
        $totalCountry = \Cache::get('totalCountry');
        if (!$totalCountry) {
            $totalCountry = Country::count('id');
            \Cache::put('totalCountry', $totalCountry);
        }
        return view('admin.country.index', compact('totalCountry'));
    }

    public function countryList(Request $request)
    {
        $countries = Country::get();
        $basicControl = basicControl();
        $siteCurrency = $basicControl->base_currency;

        return DataTables::of($countries)
            ->addColumn('checkbox', function ($item) {
                return ' <input type="checkbox" id="chk-' . $item->id . '" class="form-check-input row-tic tic-check"
                name="check" value="' . $item->id . '" data-id="' . $item->id . '">';
            })
            ->addColumn('name', function ($item) {
                $editUrl = route('admin.country.edit', $item->id);
                return '<a class="d-flex align-items-center me-2" href="' . $editUrl . '">
                            <div class="flex-shrink-0"> ' . $item->countryImage() . ' </div>
                            <div class="flex-grow-1 ms-3">
                                <h5 class="text-hover-primary mb-0">' . $item->name . '</h5>
                            </div>
                        </a>';
            })
            ->addColumn('status', function ($item) {
                return renderStatusBadge($item->status);
            })
            ->addColumn('action', function ($item) {
                $editUrl = route('admin.country.edit', $item->id);
                $state = route('admin.countryState', $item->id);
                return '<div class="btn-group" role="group">
                            <a href="' . $editUrl . '" class="btn btn-white btn-sm edit_user_btn">
                                <i class="bi-pencil-fill me-1"></i> ' . trans("Edit") . '
                            </a>
                            <div class="btn-group">
                                <button type="button" class="btn btn-white btn-icon btn-sm dropdown-toggle dropdown-toggle-empty" id="userEditDropdown" data-bs-toggle="dropdown" aria-expanded="false"></button>
                                <div class="dropdown-menu dropdown-menu-end mt-1" aria-labelledby="userEditDropdown">
                                   <a class="dropdown-item" href="' . $state . '">
                                     <i class="bi-globe dropdown-item-icon"></i> ' . trans("Manage State") . ' </a>
                                </div>
                            </div>
                        </div>';
            })->rawColumns(['action', 'checkbox', 'name', 'status'])
            ->make(true);
    }

    public function create()
    {
        return view('admin.country.create');
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'iso2' => 'required|max:2',
            'iso3' => 'required|max:3',
            'phone_code' => 'required',
        ]);

        if ($validator->fails()) {
            return back()->withInput()->withErrors($validator);
        }

        $data = Arr::except($request->all(), ['_token', '_method', 'image']);
        Country::insert($data);
        return redirect()->route('admin.country.index')->with('success', 'Country added successfully');
    }

    public function edit(Country $country)
    {
        return view('admin.country.edit', compact('country'));
    }

    public function update(Request $request, Country $country)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required',
            'iso2' => 'required|max:2',
            'iso3' => 'required|max:3',
            'phone_code' => 'required',
        ]);

        if ($validator->fails()) {
            return back()->withInput()->withErrors($validator);
        }
        $data = Arr::except($request->all(), ['_token', '_method', 'image']);
        $country->update($data);
        return redirect()->route('admin.country.index')->with('success', 'Country updated successfully');
    }

    public function importCountries()
    {
        $this->broadcastMessage('Please wait few moments...');

        if (!File::exists('assets/worldSeeder.txt')) {
            File::put('assets/worldSeeder.txt', '' . time());
            $output = new \Symfony\Component\Console\Output\BufferedOutput();
            \Illuminate\Support\Facades\Artisan::call('db:seed --class=WorldSeeder');
            // Retrieve the command output, if needed
            $output = \Illuminate\Support\Facades\Artisan::output();

            $this->broadcastMessage($output->fetch());
        }
        $this->broadcastMessage('Your function completed successfully.');
    }

    protected function broadcastMessage($message)
    {
        return response()->json(['message' => $message]);
    }

    public function deleteMultiple(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select any Country.');
            return response()->json(['error' => 1]);
        } else {
            DB::transaction(function () use ($request) {
                $countries = Country::with('states','cities')->whereIn('id', $request->strIds)->get();

                foreach ($countries as $country) {
                    $country->states->each->delete();
                    $country->cities->each->delete();
                }
                Country::whereIn('id', $request->strIds)->delete();
            });

            session()->flash('success', 'Selected Data deleted successfully');
            return response()->json(['success' => 1]);
        }
    }

    public function activeMultiple(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select any Country.');
            return response()->json(['error' => 1]);
        } else {
            Country::whereIn('id', $request->strIds)->update([
                'status' => 1,
            ]);
            session()->flash('success', 'Country Status Activated');
            return response()->json(['success' => 1]);
        }
    }

    public function inactiveMultiple(Request $request)
    {
        if ($request->strIds == null) {
            session()->flash('error', 'You do not select any Country.');
            return response()->json(['error' => 1]);
        } else {
            Country::whereIn('id', $request->strIds)->update([
                'status' => 0,
            ]);

            session()->flash('success', 'Country Status Inactive Successfully');
            return response()->json(['success' => 1]);
        }
    }


    public function countryState(Country $country)
    {
        return view('admin.country.stateList', compact('country'));
    }


}
