<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\BasicControl;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Facades\App\Services\BasicService;
use Exception;
use Illuminate\Support\Facades\Validator;
use Stevebauman\Purify\Facades\Purify;
use function GuzzleHttp\Promise\all;

class BasicControlController extends Controller
{
    public function index($settings = null)
    {
        $settings = $settings ?? 'settings';
        abort_if(!in_array($settings, array_keys(config('generalsettings'))), 404);
        $settingsDetails = config("generalsettings.{$settings}");
        return view('admin.control_panel.settings', compact('settings', 'settingsDetails'));
    }

    public function basicControl()
    {
        $data['basicControl'] = basicControl();
        $data['timeZones'] = timezone_identifiers_list();
        $data['dateFormat'] = config('dateformat');

        return view('admin.control_panel.basic_control', $data);
    }

    public function basicControlUpdate(Request $request)
    {
        $request->validate([
            'site_title' => 'required|string|min:3|max:100',
            'time_zone' => 'required|string',
            'base_currency' => 'required|string|min:1|max:100',
            'currency_symbol' => 'required|string|min:1|max:100',
            'fraction_number' => 'required|integer|not_in:0',
            'paginate' => 'required|integer|not_in:0',
            'date_format' => 'required',
            'admin_prefix' => 'required|string|min:3|max:100',
            'primary_color' => 'required|string',
            'emailThrottleTime' => 'sometimes|integer|min:1|max:60',
            'email_check_max_limit' => 'required|integer|min:1',
            'credit_bonus' => 'required|integer|min:0',
        ]);

        try {
            $basic = BasicControl();
            $response = BasicControl::updateOrCreate([
                'id' => $basic->id ?? ''
            ], [
                'site_title' => $request->site_title,
                'time_zone' => $request->time_zone,
                'base_currency' => $request->base_currency,
                'currency_symbol' => $request->currency_symbol,
                'fraction_number' => $request->fraction_number,
                'date_time_format' => $request->date_format,
                'paginate' => $request->paginate,
                'admin_prefix' => $request->admin_prefix,
                'primary_color' => $request->primary_color,
                'emailThrottleTime' => $request->emailThrottleTime,
                'emailThrottleStatus' => $request->emailThrottleStatus ?? 0,
                'email_check_max_limit' => $request->email_check_max_limit ?? 100,
                'credit_bonus' => $request->credit_bonus ?? 0,
            ]);

            if (!$response)
                throw new Exception('Something went wrong, when updating data');

            $env = [
                'APP_TIMEZONE' => $response->time_zone,
                'APP_DEBUG' => $response->error_log == 0 ? 'true' : 'false'
            ];
            BasicService::setEnv($env);

            session()->flash('success', 'Basic Control Configure Successfully');
            Artisan::call('optimize:clear');
            return back();
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }
    }

    public function basicControlActivityUpdate(Request $request)
    {

        $request->validate([
            'strong_password' => 'nullable|numeric|in:0,1',
            'registration' => 'nullable|numeric|in:0,1',
            'error_log' => 'nullable|numeric|in:0,1',
            'is_active_cron_notification' => 'nullable|numeric|in:0,1',
            'has_space_between_currency_and_amount' => 'nullable|numeric|in:0,1',
            'is_force_ssl' => 'nullable|numeric|in:0,1',
            'email_checker' => 'nullable|numeric|in:0,1',
            'is_currency_position' => 'nullable|string|in:left,right'
        ]);

        try {
            $basic = BasicControl();
            $response = BasicControl::updateOrCreate([
                'id' => $basic->id ?? ''
            ], [
                'error_log' => $request->error_log,
                'strong_password' => $request->strong_password,
                'registration' => $request->registration,
                'is_active_cron_notification' => $request->is_active_cron_notification,
                'has_space_between_currency_and_amount' => $request->has_space_between_currency_and_amount,
                'is_currency_position' => $request->is_currency_position,
                'is_force_ssl' => $request->is_force_ssl,
                'email_checker' => $request->email_checker
            ]);

            if (!$response)
                throw new Exception('Something went wrong, when updating data');

            session()->flash('success', 'Basic Control Configure Successfully');
            Artisan::call('optimize:clear');
            return back();
        } catch (\Exception $e) {
            return back()->with('error', $e->getMessage());
        }


    }

    public function openAiUpdate(Request $request)
    {
        $basicControl = basicControl();
        if ($request->isMethod('get')) {
            return view('admin.control_panel.openAi', compact('basicControl'));
        } elseif ($request->isMethod('post')) {

            $purifiedData = Purify::clean($request->all());
            $purifiedData['image'] = $request->image;
            $validator = Validator::make($purifiedData, [
                'open_ai_key' => 'required|string|min:5',
                'open_ai_model' => 'required|string|min:3',
            ]);

            if ($validator->fails()) {
                return back()->withErrors($validator)->withInput();
            }

            $purifiedData = (object)$purifiedData;
            $basicControl->open_ai_key = $purifiedData->open_ai_key;
            $basicControl->open_ai_model = $purifiedData->open_ai_model;
            $basicControl->open_ai_status = $purifiedData->open_ai_status;
            $basicControl->save();

            return back()->with('success', 'OpenAi has been updated.');
        }
    }

    public function cookieControl(Request $request)
    {
        if ($request->isMethod('get')) {
            $data['basic'] = basicControl();
            return view('admin.control_panel.cookie', $data);
        }
        elseif ($request->isMethod('post')) {
            $data = $request->validate([
                'cookie_title' => 'required|string|max:100',
                'cookie_sub_title' => 'required|string|max:100',
                'cookie_url' => 'required|string|max:100',
                'cookie_status' => 'nullable|numeric|in:0,1',
            ]);
            $basic = BasicControl();
            $basic->cookie_title = $request->cookie_title;
            $basic->cookie_sub_title = $request->cookie_sub_title;
            $basic->cookie_url = $request->cookie_url;
            $basic->cookie_status = $request->cookie_status;

            $basic->save();

            Artisan::call('optimize:clear');
            return back()->with('success', 'Successfully Updated');
        }
    }
}
